/**
 * Copyright (C) 2012 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

public class LicenseChecker {

    public interface Callback {

        public void ok();

        public void failed(Exception e);
    }

    static Consumer<String> systemBrowserUrlHandler = url -> {
        try {
            getLogger().info(
                    "Opening system browser to validate license. If the browser is not opened, please open "
                            + url + " manually");
            SystemBrowser.open(url);
        } catch (java.awt.HeadlessException e) {
            getLogger().log(Level.SEVERE, "You are running in a headless environment. Please open " + url
                    + " in any browser to validate the license.");
        } catch (IOException | URISyntaxException | UnsupportedOperationException e) {
            getLogger().log(Level.SEVERE,
                    "Error opening system browser to validate license. Please open " + url + " manually", e);
        }

    };

    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion) {
        try {
            checkLicense(productName, productVersion);
        } catch (Exception e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    public static void checkLicense(String productName, String productVersion) {
        checkLicense(productName, productVersion, systemBrowserUrlHandler);
    }

    public static void checkLicense(String productName, String productVersion, Consumer<String> noKeyUrlHandler) {
        checkLicense(new Product(productName, productVersion), noKeyUrlHandler);
    }

    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback) {
        checkLicenseAsync(productName, productVersion, callback, systemBrowserUrlHandler);
    }

    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback, Consumer<String> noKeyUrlHandler) {
        new Thread(() -> {
            try {
                checkLicense(new Product(productName, productVersion), noKeyUrlHandler);
                callback.ok();
            } catch (Exception e) {
                callback.failed(e);
            }
        }).start();
    }

    private static void checkLicense(Product product, Consumer<String> noKeyUrlHandler) {
        getLogger().fine("Checking license for " + product);
        ProKey proKey = LocalProKey.get();
        if (proKey == null) {
            // No proKey found - probably first launch
            proKey = VaadinComIntegration.openBrowserAndWaitForKey(product, noKeyUrlHandler);
            if (proKey != null) {
                LocalProKey.write(proKey);
            }
        }
        ProKeyValidator.validate(product, proKey, MachineId.get());
    }

    public static Logger getLogger() {
        return Logger.getLogger(LicenseChecker.class.getName());
    }

}
