package com.vaadin.pro.licensechecker;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.time.Instant;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.logging.Logger;

public class VaadinComIntegration {

    private static final String BASE_URL = "https://vaadin.com/pro/validate-license";
    private static final String MANUAL_PRO_KEY_URL = BASE_URL;

    public static ProKey openBrowserAndWaitForKey(Product product) {
        return openBrowserAndWaitForKey(product, LicenseChecker.systemBrowserUrlHandler);
    }

    static ProKey openBrowserAndWaitForKey(Product product, Consumer<String> noKeyUrlHandler) {
        String uniqueId = UUID.randomUUID().toString();
        String loginUrl = constructUrl(uniqueId, product);

        noKeyUrlHandler.accept(loginUrl);

        try {
            return listenForProKey(BASE_URL + "/connect/" + uniqueId);
        } catch (IOException e) {
            throw new RuntimeException(
                    "Error retreiving your subscription key from the server. Please try again or download the key manually from "
                            + MANUAL_PRO_KEY_URL,
                    e);
        }
    }

    public static String constructUrl(String uniqueId, Product product) {
        return BASE_URL + "?connect=" + uniqueId + "&productName="
                + product.getName() + "&productVersion=" + product.getVersion();
    }

    public static ProKey listenForProKey(String location) throws IOException {
        Instant timeout = Instant.now().plusSeconds(60);
        URL url = new URL(location);
        while (Instant.now().isBefore(timeout)) {
            getLogger().fine("Polling server for pro key");
            try (InputStream in = url.openStream()) {
                String data = Util.toString(in);
                return ProKey.fromJson(data);
            } catch (FileNotFoundException e) {
                // Server returns 404 if the login process has not completed
                try {
                    Thread.sleep(500);
                } catch (InterruptedException e1) {
                }
            }
        }
        return null;
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }
}
