/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.slf4j.Logger;

/**
 * Fetches an offline key from either the home directory (~/.vaadin/offlineKey)
 * or from a system property ({@code vaadin.offlineKey}) if available. Also
 * capable of writing an offline key but this should not be used except for
 * testing. Offline keys are always downloaded and stored manually.
 */
public class LocalOfflineKey {

    private static OfflineKey read(File offlineKeyLocation) throws IOException {
        if (!offlineKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(offlineKeyLocation)) {
            return OfflineKey.fromJson(Util.toString(is));
        }
    }

    private static void write(File offlineKeyLocation, OfflineKey key) throws IOException {
        offlineKeyLocation.getParentFile().mkdirs();

        try (FileOutputStream out = new FileOutputStream(offlineKeyLocation)) {
            out.write(key.toJson().getBytes(StandardCharsets.UTF_8));
        }
    }

    public static File getLocation() {
        return new File(LocalProKey.getVaadinFolder(), "offlineKey");
    }

    /**
     * Finds an offline key, either from a system property or from the home
     * directory.
     * 
     * @return an offline key or {@code null} if no key was found
     */
    public static OfflineKey get() {
        OfflineKey offlineKey = getSystemProperty();
        if (offlineKey != null) {
            getLogger().debug("Using offlineKey from system property");
            return offlineKey;
        }

        File keyLocation = getLocation();
        try {
            offlineKey = read(keyLocation);
            getLogger().debug("Found offline key in " + keyLocation);
            return offlineKey;
        } catch (IOException e) {
            getLogger().debug("Unable to read offline key", e);
            return null;
        }
    }

    private static OfflineKey getSystemProperty() {
        String json = System.getProperty("vaadin.offlineKey");
        if (json == null) {
            return null;
        }

        return OfflineKey.fromJson(json);
    }

    static void write(OfflineKey key) throws IOException {
        File keyLocation = getLocation();
        getLogger().info("Writing offline key to " + keyLocation);
        write(keyLocation, key);

    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
