/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import org.slf4j.Logger;

import elemental.json.Json;
import elemental.json.JsonException;
import elemental.json.JsonObject;

/**
 * A key for offline validation of licenses.
 * <p>
 * An offline key contains information about the user, the subscription and has
 * an expiration date. It is machine specific and tied to the value returned by
 * {@link MachineId#get()}).
 * <p>
 * A special version of an offline key is only for production builds. This key
 * is not tied to the machine id as it is meant for CI servers where the machine
 * id can change between builds. A normal offline key should always be used for
 * development.
 * <p>
 * This version has no product information in the offline key but only validates
 * that an active subscription exists.
 * <p>
 * An offline key is acquired by providing the machine id to a page on
 * vaadin.com
 */
public class OfflineKey {

    static OfflineKey fromJson(String jsonData) {
        try {
            JsonObject json = Json.parse(jsonData);
            OfflineKey proKey = new OfflineKey(json.getString("username"),
                    json.getString("proKey"), json.getString("subscription"), Long.parseLong(json.getString("expires")),
                    json.getBoolean("productionOnly"), json.getString("hash"));
            return proKey;
        } catch (JsonException e) {
            getLogger().error("Unable to parse offline key from json: " + jsonData, e);
            return null;
        }
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    private final String username;
    private final String proKey;
    private final String subscription;
    private final String hash;
    private final long expires;
    private final boolean productionOnly;

    public OfflineKey(String username, String proKey, String subscription, long expires, boolean productionOnly,
            String hash) {
        this.username = username;
        this.proKey = proKey;
        this.expires = expires;
        this.hash = hash;
        this.subscription = subscription;
        this.productionOnly = productionOnly;
    }

    public String getUsername() {
        return username;
    }

    public String getProKey() {
        return proKey;
    }

    public String getSubscription() {
        return subscription;
    }

    public long getExpires() {
        return expires;
    }

    public String getHash() {
        return hash;
    }

    public String toJson() {
        JsonObject json = Json.createObject();
        json.put("username", username);
        json.put("proKey", proKey);
        json.put("subscription", subscription);
        json.put("expires", "" + expires);
        json.put("productionOnly", productionOnly);
        json.put("hash", hash);
        return json.toJson();

    }

    public boolean isProductionOnly() {
        return productionOnly;
    }
}
