/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;

import org.slf4j.Logger;

/**
 * Validator that can allow or deny usage of a given product version for a given
 * offline key.
 * <p>
 * For development, the offline key can be used if it has not expired and the
 * checksum matches the expected for the machine it is run on.
 * <p>
 * For production, the offline key can also be used if it is a productionOnly
 * key and has not expired.
 */
public class OfflineKeyValidator {

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    public static boolean isExpired(OfflineKey key) {
        return Instant.now().isAfter(Instant.ofEpochMilli(key.getExpires()));
    }

    /**
     * Validates that the given offline license is valid and provides access to the
     * given product.
     * 
     * @param product    the product to validate
     * @param buildType  the type of build: production or development
     * @param offlineKey the offline license key
     * @param machineId  the id of the machine where we are running
     * 
     * @return {@code true} if the validation succeeded
     * @throws RuntimeException if the validation fails because and invalid offline
     *                          key was provided
     */
    public static boolean validate(Product product, BuildType buildType, OfflineKey offlineKey, String machineId) {
        if (offlineKey == null) {
            getLogger().debug("Null offlineKey");
            throw new RuntimeException(getMissingOfflineKeyMessage(machineId));
        }
        getLogger().debug("Validating offline key for " + product);
        if (History.isRecentlyValidated(product)) {
            // check only every 24h
            getLogger().debug(
                    "Skipping check as product license was recently validated.");
            return true;
        }
        String expectedHash = hash(offlineKey.getUsername(), offlineKey.getProKey(), offlineKey.getSubscription(),
                offlineKey.getExpires(), offlineKey.isProductionOnly(),
                machineId);
        if (!expectedHash.equals(offlineKey.getHash())) {
            throw new RuntimeException(getInvalidOfflineKeyMessage(machineId));
        }

        if (isExpired(offlineKey)) {
            throw new RuntimeException(getExpiredOfflineKeyMessage(machineId));
        }

        if (offlineKey.isProductionOnly() && buildType != BuildType.PRODUCTION) {
            throw new RuntimeException(getOnlyProductionMessage(machineId));
        }

        History.setLastCheckTimeNow(product);
        History.setLastSubscription(product,
                offlineKey.getSubscription());

        return true;
    }

    private static String getExpiredOfflineKeyMessage(String machineId) {
        return "Offline key has expired, " + getOfflineKeyLinkMessage(machineId);
    }

    private static String getOnlyProductionMessage(String machineId) {
        return "The provided offline key is only enabled for production builds, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getOfflineKeyLinkMessage(String machineId) {
        return "please go to https://vaadin.com/pro/validate-license?getOfflineKey="
                + machineId + " to retrieve an offline key."
                + " For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    }

    private static String getInvalidOfflineKeyMessage(String machineId) {
        return "Invalid offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    private static String getMissingOfflineKeyMessage(String machineId) {
        return "No offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String hash(String username, String proKey, String subscription, long expires, boolean productionOnly,
            String machineId) {
        String str = username + "&" + proKey + "&" + subscription + "&" + expires + "&"
                + String.valueOf(productionOnly);
        if (!productionOnly) {
            str += "&" + machineId;
        }
        return sha1(str);
    }

    private static String sha1(String str) {
        MessageDigest md = null;
        try {
            md = MessageDigest.getInstance("SHA-1");
            byte[] messageDigest = md.digest(str.getBytes());
            BigInteger no = new BigInteger(1, messageDigest);
            String hashtext = no.toString(16);
            while (hashtext.length() < 32) {
                hashtext = "0" + hashtext;
            }
            return hashtext;
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Unable to use SHA-1", e);
        }
    }

}
