/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.text.ParseException;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.jwt.SignedJWT;

/**
 * Validator that can allow or deny usage of a given product version for a given
 * offline key.
 * <p>
 * For development, the offline key can be used if it has not expired and the
 * checksum matches the expected for the machine it is run on.
 * <p>
 * For production, the offline key can also be used if it is a productionOnly
 * key and has not expired.
 */
public class OfflineKeyValidator {

    private static final Map<String, String> PUBLIC_KEYS = new HashMap<>();
    static {
        PUBLIC_KEYS.put("1", "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQBE7E9xWLjtmJZIeNhFZVPBIDwZRUI"
                + "ymfRXvkSN0h8/WEO6LzE2/eC9vlk/8E0/yPTS91f0/3Qn+BPfD3d6RjKs2kBOLpr"
                + "C4XCDrhwhLlnsncHV0oj64SspoBPWsCoeb2N7QYu/8ki5Re+/rxE/n0O5jZzruYE"
                + "JT3ZS2DVOp+PUnxuvb0=");
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    private static boolean isExpired(long expires) {
        return Instant.now().isAfter(Instant.ofEpochMilli(expires));
    }

    /**
     * Validates that the given offline license is valid and provides access to the
     * given product.
     * 
     * @param product    the product to validate
     * @param buildType  the type of build: production or development
     * @param offlineKey the offline license key
     * @param machineId  the id of the machine where we are running
     * 
     * @return {@code true} if the validation succeeded
     * @throws LicenseException if the validation fails because and invalid offline
     *                          key was provided
     */
    boolean validate(Product product, BuildType buildType, OfflineKey offlineKey, String machineId) {
        getLogger().debug("Offline validation using offlineKey for " + product);
        if (offlineKey == null) {
            getLogger().debug("No offline key found");
            return false;
        }
        if (History.isRecentlyValidated(product)) {
            // check only every 24h
            getLogger().debug(
                    "Skipping check as product license was recently validated.");
            return true;
        }

        validateOfflineKey(offlineKey, machineId);

        if (offlineKey.isProductionOnly()) {
            // Production only
            if (buildType != BuildType.PRODUCTION) {
                getLogger().debug("Offline key is only for production");
                throw new LicenseException(getOnlyProductionMessage(machineId));
            }
        } else {
            // Development
            if (!machineId.equals(offlineKey.getMachineId())) {
                getLogger().debug("Offline key has incorrect machine id");
                throw new LicenseException(getInvalidOfflineKeyMessage(machineId));
            }
        }

        if (isExpired(offlineKey.getExpires())) {
            getLogger().debug("Offline key expired");
            throw new LicenseException(getExpiredOfflineKeyMessage(machineId));
        }

        History.setLastCheckTimeNow(product);
        History.setLastSubscription(product,
                offlineKey.getSubscription());

        getLogger().debug("Offline key OK");
        return true;
    }

    private void validateOfflineKey(OfflineKey offlineKey, String machineId) {
        try {
            String jwtData = offlineKey.getJwtData();

            // Claim kid = JWT.decode(jwtData).getHeaderClaim("kid");
            SignedJWT jwt = (SignedJWT) JWTParser.parse(jwtData);
            Object kid = jwt.getHeader().getKeyID();
            KeyFactory fact = KeyFactory.getInstance("EC");
            byte[] encoded = Base64.getDecoder().decode(PUBLIC_KEYS.get(kid));
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(encoded);
            ECPublicKey publicKey = (ECPublicKey) fact.generatePublic(keySpec);

            JWSVerifier verifier = new ECDSAVerifier(publicKey);
            if (!jwt.verify(verifier)) {
                getLogger().debug("Offline key failed verification");
                throw new LicenseException(getInvalidOfflineKeyMessage(machineId));
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            getLogger().debug("Offline key could not be read", e);
            throw new LicenseException(getErrorValidatingOfflineKeyMessage(machineId), e);
        } catch (ParseException e) {
            getLogger().debug("Error parsing offline key", e);
            throw new LicenseException(getInvalidOfflineKeyMessage(machineId), e);
        } catch (JOSEException e) {
            getLogger().debug("Error reading offline key", e);
            throw new LicenseException(getInvalidOfflineKeyMessage(machineId), e);
        }

    }

    private static String getExpiredOfflineKeyMessage(String machineId) {
        return "Offline key has expired, " + getOfflineKeyLinkMessage(machineId);
    }

    private static String getOnlyProductionMessage(String machineId) {
        return "The provided offline key is only enabled for production builds, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getOfflineKeyLinkMessage(String machineId) {
        return "please go to " + getOfflineUrl(machineId) + " to retrieve an offline key."
                + " For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    }

    public static String getOfflineUrl(String machineId) {
        return "https://vaadin.com/pro/validate-license?getOfflineKey="
                + machineId;
    }

    private static String getErrorValidatingOfflineKeyMessage(String machineId) {
        return "Unable to validate offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getInvalidOfflineKeyMessage(String machineId) {
        return "Invalid offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getMissingOfflineKeyMessage(String machineId) {
        return "No offline key, " + getOfflineKeyLinkMessage(machineId);
    }

}
