package com.vaadin.uitest;

import java.io.File;
import java.util.Collections;

import com.vaadin.uitest.ai.LLMService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.ai.utils.KeysUtils;
import com.vaadin.uitest.generator.Generator;
import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.UiRoute;
import com.vaadin.uitest.parser.Parser;

/**
 * Public class to use ui-generator from other apps like copilot.
 */
public class TestCodeGenerator {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(TestCodeGenerator.class);

    public static final String AI_ASSISTANT = "AiAssistant";
    private static final Generator DEFAULT_GENERATOR = new Generator() {
    };

    public static boolean addTestDependencies(UiRoute route, String projectRoot,
            String testFolder) throws Exception {
        LOGGER.debug(
                "Checking if we need to update project test dependencies.\nPaths passed to the Generator: \n projectRoot: {}\n testFolder: {}\n route.testFile:{}\n route.testDir: {}",
                projectRoot, testFolder, route.getTestfile(),
                route.getTestsDir());
        return DEFAULT_GENERATOR.addTestDependencies(projectRoot, testFolder,
                Collections.singletonList(route),
                TestFramework.PLAYWRIGHT_JAVA);
    }

    public static File writeUiTest(UiRoute route, String source)
            throws Exception {
        LOGGER.info("Writing a '{}' test for route: '/{}' to: '{}'",
                route.getFramework(),
                route.getRoute() == null ? "" : route.getRoute(),
                route.getTestfile());
        LOGGER.info(
                "Paths passed to the Generator:\n route.testFile:{}\n route.testDir: {}",
                route.getTestfile(), route.getTestsDir());
        return DEFAULT_GENERATOR.writeTestFile(route, source);
    }

    public static void configureKeys(String aiToken, String pineKey,
            String pineUrl) {
        LOGGER.info("configuring AI Keys");
        KeysUtils.setKeys(aiToken, pineKey, pineUrl);
    }

    public static String generateTests(UiRoute route,
            TestFramework testFramework) {
        LOGGER.info("Generating a '{}' test for route '/{}' view: '{}'",
                route.getFramework(), route.getRoute(), route.getFile());
        Parser parserImplementation = getParserImplementation(AI_ASSISTANT,
                route.getBaseUrl());
        parserImplementation.generateTestScenarios(route);
        Generator generatorImplementation = getGeneratorImplementation(
                AI_ASSISTANT);
        return generatorImplementation.generateTest(route, null, testFramework);
    }

    public static String generateTests(UiRoute route) {
        return generateTests(route, null);
    }

    static public Generator getGeneratorImplementation(String name) {
        String nameSpace = Generator.class.getPackageName();
        String prefix = Generator.class.getSimpleName();
        String className = nameSpace + "." + prefix + name;
        Generator implementation = createInstance(Generator.class, className,
                null);
        LOGGER.info("Using {} Generator implementation: {}", name,
                implementation.getClass().getSimpleName());
        return implementation;
    }

    public static String implementationName(Class<?> clazz) {
        String regex = "^" + Generator.class.getSimpleName() + "|"
                + Parser.class.getSimpleName();
        return clazz.getSimpleName().replaceFirst(regex, "");
    }

    static public Parser getParserImplementation(String name, String baseUrl) {
        String nameSpace = Parser.class.getPackageName();
        String prefix = Parser.class.getSimpleName();
        String className = nameSpace + "." + prefix + name;
        Parser implementation = createInstance(Parser.class, className,
                baseUrl);
        LOGGER.info("Using {} Parser implementation: {}, baseUrl: {}", name,
                implementation.getClass().getSimpleName(), baseUrl);
        return implementation;
    }

    /**
     * Creates an instance of a class using reflection.
     *
     * @param className
     *            Fully qualified name of the class to instantiate.
     * @param interfaceClass
     *            The class it should implement, e.g., Generator.class.
     * @param param
     *            Constructor parameter (can be null for default constructor).
     * @return An instance of the class if found and instantiated, otherwise
     *         null.
     */
    public static <T> T createInstance(Class<T> interfaceClass,
            String className, Object param) {
        try {
            // Load the class dynamically
            Class<?> clazz = Class.forName(className);

            // Check if the class implements the desired interface
            if (!interfaceClass.isAssignableFrom(clazz)) {
                throw new ClassCastException("The class " + className
                        + " does not implement " + interfaceClass.getName());
            }

            if (param != null) {
                // If a parameter is provided, find the constructor that matches
                // its type
                Class<?> paramType = param.getClass();
                java.lang.reflect.Constructor<?> constructor = clazz
                        .getConstructor(paramType);

                // Create an instance with the provided parameter
                return (T) constructor.newInstance(param);
            } else {
                // If no parameter is provided, use the default constructor
                java.lang.reflect.Constructor<?> defaultConstructor = clazz
                        .getConstructor();

                // Create an instance with the default constructor
                return (T) defaultConstructor.newInstance();
            }
        } catch (ClassNotFoundException e) {
            LOGGER.error("Class {} not found.", className);
        } catch (NoSuchMethodException e) {
            LOGGER.error("No suitable constructor found for class:", className);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return null; // Return null if the class couldn't be instantiated
    }
}
