package com.vaadin.uitest.ai.prompts;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.vaadin.uitest.ai.LLMService;
import com.vaadin.uitest.parser.Parser;

public class AiPrompts {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(AiPrompts.class);

    public static class Prompts {

        private static final Map<String, Prompts> instances = new LinkedHashMap<>();

        private static ClassLoader classLoader = Thread.currentThread()
                .getContextClassLoader();

        @JsonProperty("prompts")
        private Map<String, String> prompts = new LinkedHashMap<>();

        public String get(String key) {
            return prompts.get(key);
        }

        public void load(String file) {
            Prompts p = instances.get(file);
            if (p == null) {
                InputStream stream = classLoader.getResourceAsStream(file);
                if (stream == null) {
                    LOGGER.debug("Not Loaded {}", file);
                    return;
                }
                try {
                    p = Parser.objectMapper.readValue(stream.readAllBytes(),
                            Prompts.class);
                    LOGGER.info("Loaded {}", file);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            instances.put(file, p);
            prompts.putAll(p.prompts);
        }
    }

    private final Prompts p = new Prompts();

    public AiPrompts(Class<? extends LLMService> service) {
        getAllClassNames(service).forEach(c -> {
            String srvImplementation = c
                    .replaceFirst("((Generator|Parser|)Service)$", "");
            String fileName = srvImplementation + "-prompts.yml";
            String customFileName = srvImplementation + "-custom-prompts.yml";
            p.load(fileName);
            p.load(customFileName);
        });
    }

    private List<String> getAllClassNames(Class<?> clazz) {
        List<String> names = new ArrayList<>();
        // Traverse class hierarchy
        while (clazz != null) {
            if (implementsLLMService(clazz)) {
                names.add(clazz.getSimpleName());
            }
            clazz = clazz.getSuperclass();
        }
        // Add the LLMService interface itself
        names.add(LLMService.class.getSimpleName());
        Collections.reverse(names);
        return names;
    }

    private boolean implementsLLMService(Class<?> clazz) {
        if (clazz == null) {
            return false;
        }

        // Get all interfaces implemented by the class
        Class<?>[] interfaces = clazz.getInterfaces();
        for (Class<?> iface : interfaces) {
            if (iface.equals(LLMService.class)) {
                return true;
            }
        }

        // Also check the superclasses if they implement the interface
        return implementsLLMService(clazz.getSuperclass());
    }

    public String get(String key) {
        String r = p.get(key);
        if (r == null) {
            LOGGER.error("Key '{}' not found in prompts files.", key);
        }
        return r;
    }

}
