package com.vaadin.uitest.parser;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import com.vaadin.uitest.browser.Browser;
import com.vaadin.uitest.browser.BrowserLogin;
import com.vaadin.uitest.model.UiLogin;
import com.vaadin.uitest.model.UiModel;
import com.vaadin.uitest.model.UiRoute;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public interface Parser {

    Logger LOGGER = LoggerFactory.getLogger(Parser.class);

    String TESTS_FILE = "ui-tests.yml";
    String VIEWS_FILE = "views.yml";

    ObjectMapper objectMapper = new ObjectMapper(
            YAMLFactory.builder().enable(YAMLGenerator.Feature.MINIMIZE_QUOTES)
                    .disable(YAMLGenerator.Feature.SPLIT_LINES)
                    .enable(YAMLGenerator.Feature.LITERAL_BLOCK_STYLE).build())
            .setSerializationInclusion(JsonInclude.Include.NON_DEFAULT);

    static String toJson(Object o) throws JsonProcessingException {
        return objectMapper.writerWithDefaultPrettyPrinter()
                .writeValueAsString(o);
    }

    List<UiRoute> views = new ArrayList<>();

    default void parseView(UiRoute view) {
        generateTestScenarios(view);
        updateTestScenarios(view);
    }

    void generateTestScenarios(UiRoute view);

    default UiRoute updateTestScenarios(UiRoute view) {
        return view;
    }

    default List<UiRoute> getViews() {
        return views;
    }

    default UiModel parseRoutes(File projectDir, String sourcesDir)
            throws IOException {
        views.clear();
        UiModel model = Parser.objectMapper
                .readValue(new File(projectDir, VIEWS_FILE), UiModel.class);
        views.addAll(model.getViews());
        for (UiRoute view : views) {
            String source = FileUtils.readFileToString(
                    new File(projectDir, view.getFile()), "utf-8");
            view.setSource(source);
            view.setBaseUrl(getBaseUrl());
        }
        return model;
    }

    default String doLogin(UiLogin login) {
        return doLogin(login.getLoginUrl(), getBaseUrl(),
                login.getLoginSelector(), login.getPasswordSelector(),
                login.getButtonSelector(), login.getLogin(),
                login.getPassword());
    }

    default String doLogin(String loginRoute, String baseUrl,
            String loginSelector, String passwordSelector,
            String buttonSelector, String login, String password) {
        if (loginRoute != null && baseUrl != null) {
            String url = baseUrl + "/" + loginRoute.replaceFirst("^/+", "");
            LOGGER.info("Logging in to the app by using the url={}", url);

            BrowserLogin chromeLogin = getBrowserLogin();

            chromeLogin.setParameters(url, loginSelector, passwordSelector,
                    buttonSelector, login, password);
            try {
                // perform the login process, and save the html for future
                // usage
                return chromeLogin.doLogin();
            } catch (Exception e) {
                throw new RuntimeException(e);
            } finally {
                chromeLogin.quit();
            }
        }
        return null;
    }

    default void getViewHtml(String cssSelector, String loginHtml,
            String loginRoute, UiRoute view) {
    }

    default void close() {
    }

    static boolean isValidView(UiRoute uiRoute) {
        if (uiRoute == null) {
            return false;
        }
        String prop = System.getProperty("view");
        if (prop == null || prop.isBlank()) {
            return true;
        }
        List<String> views = Arrays.asList(prop.split("[,\s]+"));
        String simpleName = uiRoute.getClassName().replaceFirst("^.*\\.", "");
        return views.contains(simpleName) || views.contains(uiRoute.getClassName())
                || views.contains(uiRoute.getRoute());
    }

    default String getBaseUrl() {
        return null;
    }

    BrowserLogin getBrowserLogin();

    Browser getBrowser();

}
