package com.vaadin.uitest.ai.utils;

import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vaadin.uitest.model.Framework;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Attribute;
import org.jsoup.nodes.Comment;
import org.jsoup.nodes.DataNode;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.select.NodeVisitor;

public class PromptUtils {

    private static String cleanHtmlCode(String htmlCode) {
        return htmlCode
                // reduce spaces tabs and NL
                .replaceAll("\\s+", " ")
                // Remove style and other unnecessary attributes
                .replaceAll(" (style|class|theme|slot|aria-\\w+)=\"[^\"]*\"",
                        "")
                // Remove extra white spaces between tags
                .replaceAll("\\s*([<>])\\s*", "$1");
    }

    public static String cleanHtml(String html) {
        String tagToRemove = "script, style, vaadin-dev-tools, vaadin-connection-indicator";
        String tagToUnwrap = "flow-container-root";
        Document document = Jsoup.parse(html);

        // remove undesired tags
        document.select(tagToRemove).remove();
        // unwrap flow container
        document.getAllElements().stream()
                .filter(e -> e.nodeName().startsWith(tagToUnwrap))
                .forEach(Element::unwrap);
        // remove attributes: style, for, generated id, empty ones
        for (Element element : document.getAllElements()) {
            element.removeAttr("style");
            element.removeAttr("for");
            if (element.id().matches(".*-\\d+$")) {
                element.removeAttr("id");
            }
            element.removeAttr("src");
            Iterator<Attribute> iterator = element.attributes().iterator();
            while (iterator.hasNext()) {
                Attribute a = iterator.next();
                if (a.getValue().isEmpty()) {
                    iterator.remove();
                }
            }
        }
        document.traverse(new NodeVisitor() {
            @Override
            public void head(Node node, int depth) {
                if (node instanceof Comment || node instanceof DataNode) {
                    node.remove();
                }
            }
        });
        return cleanHtmlCode(document.html());
    }

    public static String formatHtml(String html) {
        return Jsoup.parseBodyFragment(html).html();
    }

    public static String cleanSourceCode(String sourceCode,
            Framework framework) {
        switch (framework) {
        case FLOW -> {
            return cleanJavaCode(sourceCode);
        }
        case LIT -> {
            return cleanLitCode(sourceCode);
        }
        case REACT -> {
            return cleanReactCode(sourceCode);
        }
        }
        return sourceCode;
    }

    public static String cleanLitCode(String sourceCode) {
        return sourceCode;
    }

    public static String cleanReactCode(String sourceCode) {
        return sourceCode;
    }

    private static String removeComments(String sourceCode) {
        String commentPattern = "/\\*.*?\\*/|//[^\\n]*|(\"(?:\\\\\"|[^\"])*\")|('(?:\\\\'|[^'])*')";
        Pattern pattern = Pattern.compile(commentPattern, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(sourceCode);
        StringBuffer result = new StringBuffer();
        // If the match is not a string literal, replace comments with an empty
        // string
        while (matcher.find()) {
            if (matcher.group(1) == null && matcher.group(2) == null) {
                matcher.appendReplacement(result, "");
            }
        }
        matcher.appendTail(result);
        return result.toString()
                // reduce spaces tabs and NL
                .replace("    ", "  ").replace("\t", "  ").replaceAll("\n", "")
                .trim();
    }

    public static String cleanJavaCode(String sourceCode) {
        try {
            sourceCode = removeComments(sourceCode);
        } catch (Exception e) {
            // do nothing
        }
        return sourceCode;
    }

    // TODO implement a more stable cleaner
    public static String cleanJsSnippet(String jsSnippetFromPromptResponse) {
        jsSnippetFromPromptResponse = jsSnippetFromPromptResponse.trim();
        if (jsSnippetFromPromptResponse.startsWith("script")) {
            jsSnippetFromPromptResponse = jsSnippetFromPromptResponse
                    .replaceFirst("script", "").trim();
        }
        return jsSnippetFromPromptResponse;
    }

    // TODO implement a more stable cleaner
    public static String cleanUiAction(String actionFromPromptResponse) {
        return actionFromPromptResponse.trim();
    }
}
