package com.vaadin.uitest.generator.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class MvnUtils {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(MvnUtils.class);

    private static record NodeResult(Node node, boolean changeOccurred) {
    };

    /**
     * Adds a Maven dependency to the specified pom.xml file.
     *
     * @param pomFilePath
     *            The path to the pom.xml file.
     * @param groupId
     *            The group ID of the dependency to add.
     * @param artifactId
     *            The artifact ID of the dependency to add.
     * @param version
     *            The version of the dependency to add, or null if not
     *            specified.
     * @param scope
     *            The scope of the dependency to add, or null if not specified.
     * @return {@code true} if the pom.xml was modified, {@code false}
     *         otherwise.
     * @throws Exception
     *             If an error occurs while reading or writing the file.
     */
    public static boolean addMavenDependency(String pomFilePath, String groupId,
            String artifactId, String version, String scope) throws Exception {

        File pomFile = new File(pomFilePath);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(pomFile);

        NodeResult projectNode = getOrCreateNode(doc, doc, "project", null,
                null);
        NodeResult dependenciesNode = getOrCreateNode(doc, projectNode.node(),
                "dependencies", null, null);

        NodeResult dependencyNode = getOrCreateNode(doc,
                dependenciesNode.node(), "dependency", "groupId", groupId,
                "artifactId", artifactId);

        NodeResult versionAdded = version != null
                ? assureTagValue(doc, dependencyNode.node(), "version", version)
                : new NodeResult(null, false);

        NodeResult scopeAdded = scope != null
                ? assureTagValue(doc, dependencyNode.node(), "scope", scope)
                : new NodeResult(null, false);

        boolean changeOccured = Stream.of(projectNode, dependenciesNode,
                dependencyNode, versionAdded, scopeAdded)
                .anyMatch(n -> n.changeOccurred());

        if (changeOccured) {
            LOGGER.info("Updated pom.xml file by adding dependency: {}:{}:{}",
                    groupId, artifactId, version);
            saveProject(doc, pomFile);
        }
        return changeOccured;
    }

    /**
     * Adds the exec-maven-plugin to the specified pom.xml file for executing
     * Hilla tests. The plugin is added to the 'it' profile.
     *
     * @param pomFilePath
     *            The path to the pom.xml file.
     * @return {@code true} if the pom.xml was modified, {@code false} if the
     *         plugin was already present.
     * @throws Exception
     *             If an error occurs while reading or writing the file.
     */
    public static boolean addMavenExecPlugin(String pomFilePath)
            throws Exception {
        File pomFile = new File(pomFilePath);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(pomFile);
        Node root = doc.getDocumentElement();
        Node profilesNode = getOrCreateNode(doc, root, "profiles").node();
        Node profileNode = getOrCreateNode(doc, profilesNode, "profile", "id",
                "it").node();
        Node buildNode = getOrCreateNode(doc, profileNode, "build").node();
        Node pluginsNode = getOrCreateNode(doc, buildNode, "plugins").node();

        if (getNodes(doc, pluginsNode, "plugin", "artifactId",
                "exec-maven-plugin", null, null).size() > 0) {
            return false;
        }
        LOGGER.info(
                "Updating pom.xml file by adding the plugin org.codehaus.mojo:exec-maven-plugin to execute hilla tests");

        Node pluginNode = getOrCreateNode(doc, pluginsNode, "plugin",
                "artifactId", "exec-maven-plugin").node();
        setOrUpdateElement(doc, pluginNode, "groupId", "org.codehaus.mojo");

        Node executionsNode = getOrCreateNode(doc, pluginNode, "executions")
                .node();
        Node executionNode = getOrCreateNode(doc, executionsNode, "execution",
                "id", "run hilla tests").node();

        Node goalsNode = getOrCreateNode(doc, executionNode, "goals").node();
        setOrUpdateElement(doc, goalsNode, "goal", "exec");

        setOrUpdateElement(doc, executionNode, "phase", "integration-test");

        Node configurationNode = getOrCreateNode(doc, executionNode,
                "configuration").node();
        setOrUpdateElement(doc, configurationNode, "executable", "npm");

        Node argumentsNode = getOrCreateNode(doc, configurationNode,
                "arguments").node();
        setOrUpdateElement(doc, argumentsNode, "argument", "test");

        saveProject(doc, pomFile);
        return true;
    }

    /**
     * Adds the maven-failsafe-plugin to the specified pom.xml file. The plugin
     * is added to the 'it' profile.
     *
     * @param pomFilePath
     *            The path to the pom.xml file.
     * @return {@code true} if the pom.xml was modified, {@code false} if the
     *         plugin was already present.
     * @throws Exception
     *             If an error occurs while reading or writing the file.
     */
    public static boolean addMavenFailsafePlugin(String pomFilePath)
            throws Exception {
        File pomFile = new File(pomFilePath);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(pomFile);
        Node root = doc.getDocumentElement();
        Node profilesNode = getOrCreateNode(doc, root, "profiles").node();
        Node profileNode = getOrCreateNode(doc, profilesNode, "profile", "id",
                "it").node();
        Node buildNode = getOrCreateNode(doc, profileNode, "build").node();
        Node pluginsNode = getOrCreateNode(doc, buildNode, "plugins").node();

        if (getNodes(doc, pluginsNode, "plugin", "artifactId",
                "maven-failsafe-plugin", null, null).size() > 0) {
            return false;
        }
        LOGGER.info(
                "Updating pom.xml file by adding the plugin org.apache.maven.plugins:exec-maven-plugin to execute hilla tests");

        Node pluginNode = getOrCreateNode(doc, pluginsNode, "plugin",
                "artifactId", "maven-failsafe-plugin").node();
        setOrUpdateElement(doc, pluginNode, "groupId",
                "org.apache.maven.plugins");

        Node executionsNode = getOrCreateNode(doc, pluginNode, "executions")
                .node();
        Node executionNode = getOrCreateNode(doc, executionsNode, "execution")
                .node();

        Node goalsNode = getOrCreateNode(doc, executionNode, "goals").node();
        setOrUpdateElement(doc, goalsNode, "goal", "integration-test");
        setOrUpdateElement(doc, goalsNode, "goal", "verify");

        Node configurationNode = getOrCreateNode(doc, executionNode,
                "configuration").node();
        setOrUpdateElement(doc, configurationNode, "trimStackTrace", "false");
        setOrUpdateElement(doc, configurationNode, "enableAssertions", "true");

        saveProject(doc, pomFile);
        return true;
    }

    /**
     * Saves the XML document to the specified file with proper formatting.
     *
     * @param doc
     *            The Document to save.
     * @param file
     *            The File to which the document should be saved.
     * @throws FileNotFoundException
     *             If the file cannot be found or created.
     * @throws TransformerException
     *             If an error occurs during the transformation or writing
     *             process.
     */
    private static void saveProject(Document doc, File file)
            throws FileNotFoundException, TransformerException {

        TransformerFactory transformerFactory = TransformerFactory
                .newInstance();
        Transformer transformer = transformerFactory
                .newTransformer(new StreamSource(IOUtils.toInputStream(""
                        + "<xsl:stylesheet version=\"1.0\" xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\">\n"
                        + "  <xsl:output indent=\"yes\"/>\n"
                        + "  <xsl:strip-space elements=\"*\"/>\n"
                        + "  <xsl:template match=\"@*|node()\">\n"
                        + "    <xsl:copy>\n"
                        + "      <xsl:apply-templates select=\"@*|node()\"/>\n"
                        + "    </xsl:copy>\n" + "  </xsl:template>\n"
                        + "</xsl:stylesheet>\n" + "", "UTF-8")));
        DOMSource source = new DOMSource(doc);
        StreamResult result = new StreamResult(new FileOutputStream(file));
        transformer.transform(source, result);
    }

    /**
     * Sets or updates an element with the given tag name and text content under
     * the parent node. If the element already exists, its content is updated.
     * Otherwise, a new element is created.
     *
     * @param doc
     *            The Document used for creating new elements.
     * @param parentNode
     *            The parent Node where the element should be set or updated.
     * @param tagName
     *            The name of the element to set or update.
     * @param textContent
     *            The text content to set for the element.
     */
    private static void setOrUpdateElement(Document doc, Node parentNode,
            String tagName, String textContent) {
        NodeList nodes = ((Element) parentNode).getElementsByTagName(tagName);
        if (nodes.getLength() > 0) {
            nodes.item(0).setTextContent(textContent);
        } else {
            Element elem = doc.createElement(tagName);
            elem.setTextContent(textContent);
            parentNode.appendChild(elem);
        }
    }

    /**
     * Retrieves nodes with a specific name from the parent node that match the
     * given tag criteria.
     *
     * @param doc
     *            The Document containing the nodes.
     * @param parentNode
     *            The parent Node to search in.
     * @param name
     *            The name of the nodes to find.
     * @param tag
     *            Optional tag name to filter nodes (can be null).
     * @param value
     *            Optional value for the tag to filter nodes (can be null).
     * @param tag2
     *            Optional second tag name to filter nodes (can be null).
     * @param value2
     *            Optional value for the second tag to filter nodes (can be
     *            null).
     * @return A List of Nodes matching the criteria.
     */
    private static List<Node> getNodes(Document doc, Node parentNode,
            String name, String tag, String value, String tag2, String value2) {
        List<Node> ret = new ArrayList<Node>();
        NodeList children = parentNode.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (name.equals(child.getNodeName())) {
                if (tag == null || value == null) {
                    ret.add(child);
                    continue;
                }
                if (child.getNodeType() == Node.ELEMENT_NODE) {
                    Element elm = (Element) child;
                    NodeList list = elm.getElementsByTagName(tag);
                    if (list.getLength() > 0) {
                        String curr = list.item(0).getTextContent().trim();
                        if (curr.equals(value)) {
                            ret.add(child);
                        }
                    }
                }
            }
        }
        if (tag2 != null && value2 != null) {
            ret = ret.stream()
                    .filter(n -> hasTagValue(doc, parentNode, tag2, value2))
                    .collect(Collectors.toList());
        }
        return ret;
    }

    /**
     * Gets an existing node with the specified name or creates a new one if it
     * doesn't exist.
     *
     * @param doc
     *            The Document used for creating new elements.
     * @param parentNode
     *            The parent Node where to look for or create the node.
     * @param name
     *            The name of the node to get or create.
     * @return A NodeResult containing the node and a flag indicating if a new
     *         node was created.
     */
    private static NodeResult getOrCreateNode(Document doc, Node parentNode,
            String name) {
        return getOrCreateNode(doc, parentNode, name, null, null, null, null);
    }

    /**
     * Gets an existing node with the specified name and tag value or creates a
     * new one if it doesn't exist.
     *
     * @param doc
     *            The Document used for creating new elements.
     * @param parentNode
     *            The parent Node where to look for or create the node.
     * @param name
     *            The name of the node to get or create.
     * @param tag
     *            The tag name to filter or add to the created node.
     * @param value
     *            The value for the tag.
     * @return A NodeResult containing the node and a flag indicating if a new
     *         node was created.
     */
    private static NodeResult getOrCreateNode(Document doc, Node parentNode,
            String name, String tag, String value) {
        return getOrCreateNode(doc, parentNode, name, tag, value, null, null);
    }

    /**
     * Gets an existing node with the specified name and tag values or creates a
     * new one if it doesn't exist.
     *
     * @param doc
     *            The Document used for creating new elements.
     * @param parentNode
     *            The parent Node where to look for or create the node.
     * @param name
     *            The name of the node to get or create.
     * @param tag
     *            The first tag name to filter or add to the created node.
     * @param value
     *            The value for the first tag.
     * @param tag2
     *            The second tag name to filter or add to the created node.
     * @param value2
     *            The value for the second tag.
     * @return A NodeResult containing the node and a flag indicating if a new
     *         node was created.
     */
    private static NodeResult getOrCreateNode(Document doc, Node parentNode,
            String name, String tag, String value, String tag2, String value2) {
        List<Node> nodes = getNodes(doc, parentNode, name, tag, value, tag2,
                value2);
        if (nodes.size() > 0) {
            return new NodeResult(nodes.get(0), false);
        }
        Element elem = doc.createElement(name);
        parentNode.appendChild(elem);
        if (tag != null && value != null) {
            Element tagElem = doc.createElement(tag);
            tagElem.setTextContent(value);
            elem.appendChild(tagElem);
        }
        if (tag2 != null && value2 != null) {
            Element tagElem = doc.createElement(tag2);
            tagElem.setTextContent(value2);
            elem.appendChild(tagElem);
        }
        return new NodeResult(elem, true);
    }

    /**
     * Checks if a specified tag in an XML node has a specified value.
     *
     * @param doc
     *            The Document containing the node (not used in the current
     *            implementation)
     * @param node
     *            The parent Node to search in
     * @param tag
     *            The tag name to look for
     * @param value
     *            The expected value of the tag
     * @return true if a child element with the specified tag name has the
     *         specified value, false otherwise
     */
    private static boolean hasTagValue(Document doc, Node node, String tag,
            String value) {
        NodeList children = node.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (child.getNodeType() == Node.ELEMENT_NODE) {
                Element elm = (Element) child;
                NodeList list = elm.getElementsByTagName(tag);
                if (list.getLength() > 0) {
                    String curr = list.item(0).getTextContent().trim();
                    if (curr.equals(value)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Assures that a given XML tag within a node has a specific value. If the
     * tag exists and has the correct value, no changes are made. If the tag
     * exists but has a different value, the tag's value is updated. If the tag
     * does not exist, it is created and appended to the node with the specified
     * value.
     *
     * @param doc
     *            The XML document to which the node belongs. Used for creating
     *            new elements.
     * @param node
     *            The parent node to check for the tag.
     * @param tag
     *            The name of the XML tag to assure.
     * @param value
     *            The desired value for the XML tag.
     * @return A {@link NodeResult} object indicating whether the node was
     *         modified and the node itself.
     */
    private static NodeResult assureTagValue(Document doc, Node node,
            String tag, String value) {
        NodeList children = node.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (child.getNodeType() == Node.ELEMENT_NODE) {
                if (child.getNodeName().equals(tag)) {
                    if (!child.getTextContent().trim().equals(value)) {
                        child.setNodeValue(value);
                        return new NodeResult(node, true);
                    }
                    return new NodeResult(node, false);
                }

            }
        }
        Element tagElem = doc.createElement(tag);
        tagElem.setTextContent(value);
        node.appendChild(tagElem);
        return new NodeResult(node, true);
    }

    /**
     * Checks if the given pom.xml file contains a non-empty list of dependency
     * nodes inside the {@literal <dependencyManagement>} section.
     *
     * @param pathToPom
     *            The file path to the
     *
     *            <pre>
     *            pom.xml
     *            </pre>
     *
     *            .
     * @return {@code true} if the {@literal <dependencyManagement>} section
     *         exists and contains at least one dependency; {@code false}
     *         otherwise.
     */
    public static boolean isDependencyManagementPresent(String pathToPom) {
        try {
            File pomFile = new File(pathToPom);
            DocumentBuilderFactory factory = DocumentBuilderFactory
                    .newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(pomFile);

            NodeList dependencyManagementList = doc
                    .getElementsByTagName("dependencyManagement");
            if (dependencyManagementList.getLength() == 0) {
                return false;
            }

            Node dependencyManagementNode = dependencyManagementList.item(0);
            NodeList dependenciesList = ((Element) dependencyManagementNode)
                    .getElementsByTagName("dependencies");

            if (dependenciesList.getLength() == 0) {
                return false;
            }

            Node dependenciesNode = dependenciesList.item(0);
            NodeList dependencyList = ((Element) dependenciesNode)
                    .getElementsByTagName("dependency");

            return dependencyList.getLength() > 0;

        } catch (Exception e) {
            LOGGER.error("Failed to read pom file", e);
            return false;
        }
    }

    public static boolean isSpringBootProject(String path) throws Exception {
        return hasExpectedParent(path, "org.springframework.boot",
                "spring-boot-starter-parent");
    }

    private static boolean hasExpectedParent(String pomFilePath,
            String expectedGroupId, String expectedArtifactId)
            throws Exception {

        File pomFile = new File(pomFilePath);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(pomFile);

        NodeList parentNodes = doc.getElementsByTagName("parent");
        if (parentNodes.getLength() == 0) {
            return false; // No <parent> tag present
        }

        Element parentElement = (Element) parentNodes.item(0);
        String groupId = getTextContentOfTag(parentElement, "groupId");
        String artifactId = getTextContentOfTag(parentElement, "artifactId");

        return expectedGroupId.equals(groupId)
                && expectedArtifactId.equals(artifactId);
    }

    private static String getTextContentOfTag(Element parent, String tagName) {
        NodeList list = parent.getElementsByTagName(tagName);
        if (list.getLength() > 0) {
            return list.item(0).getTextContent().trim();
        }
        return null;
    }

    public static boolean addSpringBootPluginToItProfile(String pomFilePath)
            throws Exception {
        File pomFile = new File(pomFilePath);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(pomFile);
        Node root = doc.getDocumentElement();

        Node profilesNode = getOrCreateNode(doc, root, "profiles").node();
        Node profileNode = getOrCreateNode(doc, profilesNode, "profile", "id",
                "it").node();
        Node buildNode = getOrCreateNode(doc, profileNode, "build").node();
        Node pluginsNode = getOrCreateNode(doc, buildNode, "plugins").node();

        if (getNodes(doc, pluginsNode, "plugin", "artifactId",
                "spring-boot-maven-plugin", null, null).size() > 0) {
            return false;
        }

        LOGGER.info("Adding spring-boot-maven-plugin to IT profile");

        Node springBootPlugin = getOrCreateNode(doc, pluginsNode, "plugin",
                "artifactId", "spring-boot-maven-plugin").node();
        setOrUpdateElement(doc, springBootPlugin, "groupId",
                "org.springframework.boot");

        Node executionsNode = getOrCreateNode(doc, springBootPlugin,
                "executions").node();

        createExecution(doc, executionsNode, "start-spring-boot",
                "pre-integration-test", "start");
        createExecution(doc, executionsNode, "stop-spring-boot",
                "post-integration-test", "stop");

        saveProject(doc, pomFile);
        return true;
    }

    private static Node createExecution(Document doc, Node executionsNode,
            String id, String phase, String goal) {

        Node executionNode = doc.createElement("execution");
        executionsNode.appendChild(executionNode);

        setOrUpdateElement(doc, executionNode, "id", id);
        setOrUpdateElement(doc, executionNode, "phase", phase);

        Node goalsNode = doc.createElement("goals");
        executionNode.appendChild(goalsNode);

        Element goalElement = doc.createElement("goal");
        goalElement.setTextContent(goal);
        goalsNode.appendChild(goalElement);

        return executionNode;
    }

}