/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.ui;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.classic.v8.event.LayoutEvents;
import com.vaadin.classic.v8.server.Helpers;
import com.vaadin.classic.v8.server.Sizeable;
import com.vaadin.classic.v8.shared.AbstractComponentState;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.shared.Registration;

/**
 * Classic Component implementation of V8's com.vaadin.ui.AbsoluteLayout.
 * <p>
 * It is a layout implementation that mimics html absolute positioning.
 * </p>
 */
@Tag(Tag.DIV)
@CssImport("./absolute-layout.css")
public class AbsoluteLayout extends AbstractLayout implements LayoutEvents.LayoutClickNotifier {
    /**
     * Class name, prefix in styling.
     */
    public static final String CLASSNAME = "v-absolutelayout";

    /**
     * A list of all wrappers this layout contains.
     */
    private final LinkedHashSet<Wrapper> children = new LinkedHashSet<>();

    /**
     * Creates an AbsoluteLayout with full size.
     */
    public AbsoluteLayout() {
        setSizeFull();
        setPrimaryStyleName(CLASSNAME);
        setStyleName("v-layout");
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.ui.AbstractComponentContainer#addComponent(com.vaadin.ui.
     * Component )
     */
    @Override
    public void addComponent(Component c) {
        addComponent(c, new ComponentPosition());
    }

    /**
     * Adds a component to the layout. The component can be positioned by
     * providing a string formatted in CSS-format.
     * <p>
     * For example the string "top:10px;left:10px" will position the component
     * 10 pixels from the left and 10 pixels from the top. The identifiers:
     * "top","left","right" and "bottom" can be used to specify the position.
     * </p>
     *
     * @param c           The component to add to the layout
     * @param cssPosition The css position string
     */
    public void addComponent(Component c, String cssPosition) {
        ComponentPosition position = new ComponentPosition();
        position.setCSSString(cssPosition);
        addComponent(c, position);
    }

    /**
     * Adds the component using the given position. Ensures the position is only
     * set if the component is added correctly.
     *
     * @param c        The component to add
     * @param position The position info for the component. Must not be null.
     * @throws IllegalArgumentException If adding the component failed
     */
    private void addComponent(Component c, ComponentPosition position)
            throws IllegalArgumentException {
        // layout -> wrapper -> component
        boolean isChildOfThis = c.getParent()
                .flatMap(Component::getParent)
                .map(this::equals)
                .orElse(false);
        if (isChildOfThis) {
            removeComponent(c);
        }
        /*
         * Create position instance and add it to componentToCoordinates map. We
         * need to do this before we call addComponent so the attachListeners
         * can access this position. #6368
         */
        Wrapper wrapper = new Wrapper(c, position);
        children.add(wrapper);
        markAsDirty();
        try {
            super.addComponent(wrapper);
        } catch (IllegalArgumentException e) {
            children.remove(wrapper);
            throw e;
        }
    }

    @Override
    public void removeComponent(Component component) {
        Optional<Wrapper> childOptional = children.stream()
                .filter(child -> child.getContent().equals(component))
                .findFirst();
        if (childOptional.isPresent()) {
            Wrapper wrapper = childOptional.orElseThrow();
            children.remove(wrapper);
            super.removeComponent(wrapper);
        }
    }

    @Override
    public void replaceComponent(Component oldComponent, Component newComponent) {
        ComponentPosition position = getPosition(oldComponent);
        removeComponent(oldComponent);
        addComponent(newComponent, position);
    }

    @Override
    public int getComponentCount() {
        return children.size();
    }

    @Override
    public Iterator<Component> iterator() {
        return children.stream()
                .map(Wrapper::getContent)
                .collect(Collectors.toUnmodifiableList())
                .iterator();
    }

    /**
     * Gets the position of a component in the layout. Returns null if component
     * is not attached to the layout.
     * <p>
     * Note that you cannot update the position by updating this object. Call
     * {@link #setPosition(Component, ComponentPosition)} with the updated
     * {@link ComponentPosition} object.
     * </p>
     *
     * @param component The component which position is needed
     * @return An instance of ComponentPosition containing the position of the
     * component, or null if the component is not enclosed in the
     * layout.
     */
    public ComponentPosition getPosition(Component component) {
        return children.stream()
                .filter(wrapper -> wrapper.getContent().equals(component))
                .findFirst()
                .map(Wrapper::getPosition)
                .orElse(null);
    }

    /**
     * Sets the position of a component in the layout.
     *
     * @param component the component to modify
     * @param position  the new position of the component
     */
    public void setPosition(Component component, ComponentPosition position) {
        Wrapper wrapper = children.stream()
                .filter(c -> c.getContent().equals(component))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("Component must be a child of this layout"));
        wrapper.setPosition(position);
        markAsDirty();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/classic-component-pack for
     * mitigation options.
     *
     * @deprecated
     */
    @Override
    public Registration addLayoutClickListener(LayoutEvents.LayoutClickListener listener) {
        return addListener("lClick", LayoutEvents.LayoutClickEvent.class, null, null);
    }

    @Deprecated
    public void removeLayoutClickListener(LayoutEvents.LayoutClickListener listener) {
        Helpers.logUnsupportedApiCall(getClass(), "removeLayoutClickListener");
    }

    /**
     * The ComponentPosition class represents a component's position within the
     * absolute layout. It contains the attributes for left, right, top and
     * bottom and the units used to specify them.
     */
    public class ComponentPosition implements Serializable {

        private int zIndex = -1;
        private Float topValue = null;
        private Float rightValue = null;
        private Float bottomValue = null;
        private Float leftValue = null;

        private Unit topUnits = Unit.PIXELS;
        private Unit rightUnits = Unit.PIXELS;
        private Unit bottomUnits = Unit.PIXELS;
        private Unit leftUnits = Unit.PIXELS;

        /**
         * Sets the position attributes using CSS syntax. Attributes not
         * included in the string are reset to their unset states.
         *
         * <code><pre>
         * setCSSString("top:10px;left:20%;z-index:16;");
         * </pre></code>
         *
         * @param css inline css to set the position
         */
        public void setCSSString(String css) {
            topValue = rightValue = bottomValue = leftValue = null;
            topUnits = rightUnits = bottomUnits = leftUnits = Unit.PIXELS;
            zIndex = -1;
            if (css == null) {
                return;
            }

            for (String cssProperty : css.split(";")) {
                String[] keyValuePair = cssProperty.split(":");
                String key = keyValuePair[0].trim();
                if (key.isEmpty()) {
                    continue;
                }
                if (key.equals("z-index")) {
                    zIndex = Integer.parseInt(keyValuePair[1].trim());
                } else {
                    String value;
                    if (keyValuePair.length > 1) {
                        value = keyValuePair[1].trim();
                    } else {
                        value = "";
                    }
                    String symbol = value.replaceAll("[0-9\\.\\-]+", "");
                    if (!symbol.isEmpty()) {
                        value = value.substring(0, value.indexOf(symbol))
                                .trim();
                    }
                    float v = Float.parseFloat(value);
                    Unit unit = Unit.getUnitFromSymbol(symbol);
                    if (key.equals("top")) {
                        topValue = v;
                        topUnits = unit;
                    } else if (key.equals("right")) {
                        rightValue = v;
                        rightUnits = unit;
                    } else if (key.equals("bottom")) {
                        bottomValue = v;
                        bottomUnits = unit;
                    } else if (key.equals("left")) {
                        leftValue = v;
                        leftUnits = unit;
                    }
                }
            }
            markAsDirty();
        }

        /**
         * Converts the internal values into a valid CSS string.
         *
         * @return A valid CSS string
         */
        public String getCSSString() {
            String s = "";
            if (topValue != null) {
                s += "top:" + topValue + topUnits.getSymbol() + ";";
            }
            if (rightValue != null) {
                s += "right:" + rightValue + rightUnits.getSymbol() + ";";
            }
            if (bottomValue != null) {
                s += "bottom:" + bottomValue + bottomUnits.getSymbol() + ";";
            }
            if (leftValue != null) {
                s += "left:" + leftValue + leftUnits.getSymbol() + ";";
            }
            if (zIndex >= 0) {
                s += "z-index:" + zIndex + ";";
            }
            return s;
        }

        /**
         * Sets the 'top' attribute; distance from the top of the component to
         * the top edge of the layout.
         *
         * @param topValue The value of the 'top' attribute
         * @param topUnits The unit of the 'top' attribute. See UNIT_SYMBOLS for a
         *                 description of the available units.
         */
        public void setTop(Float topValue, Unit topUnits) {
            this.topValue = topValue;
            this.topUnits = topUnits;
            markAsDirty();
        }

        /**
         * Sets the 'right' attribute; distance from the right of the component
         * to the right edge of the layout.
         *
         * @param rightValue The value of the 'right' attribute
         * @param rightUnits The unit of the 'right' attribute. See UNIT_SYMBOLS for a
         *                   description of the available units.
         */
        public void setRight(Float rightValue, Unit rightUnits) {
            this.rightValue = rightValue;
            this.rightUnits = rightUnits;
            markAsDirty();
        }

        /**
         * Sets the 'bottom' attribute; distance from the bottom of the
         * component to the bottom edge of the layout.
         *
         * @param bottomValue The value of the 'bottom' attribute
         * @param bottomUnits The unit of the 'bottom' attribute. See UNIT_SYMBOLS for a
         *                    description of the available units.
         */
        public void setBottom(Float bottomValue, Unit bottomUnits) {
            this.bottomValue = bottomValue;
            this.bottomUnits = bottomUnits;
            markAsDirty();
        }

        /**
         * Sets the 'left' attribute; distance from the left of the component to
         * the left edge of the layout.
         *
         * @param leftValue The value of the 'left' attribute
         * @param leftUnits The unit of the 'left' attribute. See UNIT_SYMBOLS for a
         *                  description of the available units.
         */
        public void setLeft(Float leftValue, Unit leftUnits) {
            this.leftValue = leftValue;
            this.leftUnits = leftUnits;
            markAsDirty();
        }

        /**
         * Sets the 'z-index' attribute; the visual stacking order.
         *
         * @param zIndex The z-index for the component.
         */
        public void setZIndex(int zIndex) {
            this.zIndex = zIndex;
            markAsDirty();
        }

        /**
         * Sets the value of the 'top' attribute; distance from the top of the
         * component to the top edge of the layout.
         *
         * @param topValue The value of the 'left' attribute
         */
        public void setTopValue(Float topValue) {
            this.topValue = topValue;
            markAsDirty();
        }

        /**
         * Gets the 'top' attributes value in current units.
         *
         * @return The value of the 'top' attribute, null if not set
         * @see #getTopUnits()
         */
        public Float getTopValue() {
            return topValue;
        }

        /**
         * Gets the 'right' attributes value in current units.
         *
         * @return The value of the 'right' attribute, null if not set
         * @see #getRightUnits()
         */
        public Float getRightValue() {
            return rightValue;
        }

        /**
         * Sets the 'right' attribute value (distance from the right of the
         * component to the right edge of the layout). Currently active units
         * are maintained.
         *
         * @param rightValue The value of the 'right' attribute
         * @see #setRightUnits(Unit)
         */
        public void setRightValue(Float rightValue) {
            this.rightValue = rightValue;
            markAsDirty();
        }

        /**
         * Gets the 'bottom' attributes value using current units.
         *
         * @return The value of the 'bottom' attribute, null if not set
         * @see #getBottomUnits()
         */
        public Float getBottomValue() {
            return bottomValue;
        }

        /**
         * Sets the 'bottom' attribute value (distance from the bottom of the
         * component to the bottom edge of the layout). Currently active units
         * are maintained.
         *
         * @param bottomValue The value of the 'bottom' attribute
         * @see #setBottomUnits(Unit)
         */
        public void setBottomValue(Float bottomValue) {
            this.bottomValue = bottomValue;
            markAsDirty();
        }

        /**
         * Gets the 'left' attributes value using current units.
         *
         * @return The value of the 'left' attribute, null if not set
         * @see #getLeftUnits()
         */
        public Float getLeftValue() {
            return leftValue;
        }

        /**
         * Sets the 'left' attribute value (distance from the left of the
         * component to the left edge of the layout). Currently active units are
         * maintained.
         *
         * @param leftValue The value of the 'left' CSS-attribute
         * @see #setLeftUnits(Unit)
         */
        public void setLeftValue(Float leftValue) {
            this.leftValue = leftValue;
            markAsDirty();
        }

        /**
         * Gets the unit for the 'top' attribute.
         *
         * @return See {@link Sizeable} UNIT_SYMBOLS for a description of the
         * available units.
         */
        public Unit getTopUnits() {
            return topUnits;
        }

        /**
         * Sets the unit for the 'top' attribute.
         *
         * @param topUnits See {@link Sizeable} UNIT_SYMBOLS for a description of the
         *                 available units.
         */
        public void setTopUnits(Unit topUnits) {
            this.topUnits = topUnits;
            markAsDirty();
        }

        /**
         * Gets the unit for the 'right' attribute.
         *
         * @return See {@link Sizeable} UNIT_SYMBOLS for a description of the
         * available units.
         */
        public Unit getRightUnits() {
            return rightUnits;
        }

        /**
         * Sets the unit for the 'right' attribute.
         *
         * @param rightUnits See {@link Sizeable} UNIT_SYMBOLS for a description of the
         *                   available units.
         */
        public void setRightUnits(Unit rightUnits) {
            this.rightUnits = rightUnits;
            markAsDirty();
        }

        /**
         * Gets the unit for the 'bottom' attribute.
         *
         * @return See {@link Sizeable} UNIT_SYMBOLS for a description of the
         * available units.
         */
        public Unit getBottomUnits() {
            return bottomUnits;
        }

        /**
         * Sets the unit for the 'bottom' attribute.
         *
         * @param bottomUnits See {@link Sizeable} UNIT_SYMBOLS for a description of the
         *                    available units.
         */
        public void setBottomUnits(Unit bottomUnits) {
            this.bottomUnits = bottomUnits;
            markAsDirty();
        }

        /**
         * Gets the unit for the 'left' attribute.
         *
         * @return See {@link Sizeable} UNIT_SYMBOLS for a description of the
         * available units.
         */
        public Unit getLeftUnits() {
            return leftUnits;
        }

        /**
         * Sets the unit for the 'left' attribute.
         *
         * @param leftUnits See {@link Sizeable} UNIT_SYMBOLS for a description of the
         *                  available units.
         */
        public void setLeftUnits(Unit leftUnits) {
            this.leftUnits = leftUnits;
            markAsDirty();
        }

        /**
         * Gets the 'z-index' attribute.
         *
         * @return the zIndex The z-index attribute
         */
        public int getZIndex() {
            return zIndex;
        }

        /*
         * (non-Javadoc)
         *
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return getCSSString();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            ComponentPosition position = (ComponentPosition) o;
            return zIndex == position.zIndex && Objects.equals(topValue, position.topValue) && Objects.equals(rightValue, position.rightValue) && Objects.equals(bottomValue, position.bottomValue) && Objects.equals(leftValue, position.leftValue) && topUnits == position.topUnits && rightUnits == position.rightUnits && bottomUnits == position.bottomUnits && leftUnits == position.leftUnits;
        }

        @Override
        public int hashCode() {
            return Objects.hash(zIndex, topValue, rightValue, bottomValue, leftValue, topUnits, rightUnits, bottomUnits, leftUnits);
        }
    }

    /**
     * Wrapper for any child component of an AbsoluteLayout. It holds the
     * position of the component.
     */
    @Tag(Tag.DIV)
    private class Wrapper extends AbstractSingleComponentContainer {

        private String css;

        private ComponentPosition position;

        Wrapper(Component content, ComponentPosition position) {
            addStyleName(CLASSNAME + "-wrapper");
            addStyleName(CLASSNAME + "-wrapper-absolutelayout-childcomponent");
            setContent(content);
            this.setPosition(position);
        }

        void setPosition(String cssPosition) {
            if (css == null || !css.equals(cssPosition)) {
                position.setCSSString(cssPosition);
                css = cssPosition;
                String top, right, bottom, left, zIndex;
                top = right = bottom = left = zIndex = null;
                if (!css.isEmpty()) {
                    for (String property : css.split(";")) {
                        String[] keyValue = property.split(":");
                        String key = keyValue[0];
                        String value = keyValue[1];
                        switch (key) {
                            case "left":
                                left = value;
                                break;
                            case "top":
                                top = value;
                                break;
                            case "right":
                                right = value;
                                break;
                            case "bottom":
                                bottom = value;
                                break;
                            case "z-index":
                                zIndex = value;
                                break;
                        }
                    }
                }

                this.getElement().getStyle()
                        .set("z-index", zIndex)
                        .set("top", top)
                        .set("left", left)
                        .set("right", right)
                        .set("bottom", bottom);
            }
//            updateCaptionPosition(); TODO
        }

        ComponentPosition getPosition() {
            return position;
        }

        void setPosition(ComponentPosition position) {
            this.position = position;
            this.setPosition(position.getCSSString());
        }
    }
}
