/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.StringTokenizer;
import java.util.stream.Collectors;

import com.vaadin.classic.v8.event.ActionManager;
import com.vaadin.classic.v8.event.ShortcutListener;
import com.vaadin.classic.v8.server.AbstractClientConnector;
import com.vaadin.classic.v8.server.ComponentSizeValidator;
import com.vaadin.classic.v8.server.ErrorMessage;
import com.vaadin.classic.v8.server.Helpers;
import com.vaadin.classic.v8.server.Resource;
import com.vaadin.classic.v8.server.SizeWithUnit;
import com.vaadin.classic.v8.server.Sizeable;
import com.vaadin.classic.v8.shared.AbstractComponentState;
import com.vaadin.classic.v8.shared.ui.ContentMode;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.HasEnabled;
import com.vaadin.flow.data.binder.ErrorLevel;
import com.vaadin.flow.dom.ElementConstants;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.Registration;

/**
 * An abstract base class for most legacy components. In addition to its own
 * legacy API, introduces the API from the legacy {@code Component} interface in
 * framework 8.
 */
public abstract class AbstractComponent extends AbstractClientConnector
        implements Sizeable, HasEnabled {

    private static final String HAS_HEIGHT_STYLE = "v-has-height";
    private static final String HAS_WIDTH_STYLE = "v-has-width";
    private SizeWithUnit cachedWidth;
    private SizeWithUnit cachedHeight;
    private String primaryStyleName;
    private List<String> customStyles;
    private List<String> internalStyles;
    private Locale locale;

    public AbstractComponent() {
        internalStyles = new ArrayList<>();
        internalStyles.add("v-widget");
        internalStyles.add("v-lcp");
    }

    @Override
    public void beforeClientResponse(boolean initial) {
        super.beforeClientResponse(initial);

        // legacy FW magic that prevents setting percentage size in some cases
        if (getHeight() >= 0 && (getHeightUnits() != Unit.PERCENTAGE
                || ComponentSizeValidator.parentCanDefineHeight(this))) {
            getElement().getStyle().set(ElementConstants.STYLE_HEIGHT,
                    getCSSHeight());
            addInternalStyles(HAS_HEIGHT_STYLE);
        } else {
            getElement().getStyle().set(ElementConstants.STYLE_HEIGHT, null);
            removeInternalStyles(HAS_HEIGHT_STYLE);
        }

        if (getWidth() >= 0 && (getWidthUnits() != Unit.PERCENTAGE
                || ComponentSizeValidator.parentCanDefineWidth(this))) {
            getElement().getStyle().set(ElementConstants.STYLE_WIDTH,
                    getCSSWidth());
            addInternalStyles(HAS_WIDTH_STYLE);
        } else {
            getElement().getStyle().set(ElementConstants.STYLE_WIDTH, null);
            removeInternalStyles(HAS_WIDTH_STYLE);
        }

        final String styleName = createClassName();
        getElement().setAttribute("class", styleName);
    }

    /**
     * Sets the parent classic component of the component. The role of this
     * method is to just make sure the old parent is flagged as dirty so that
     * changes are applied (e.g. to sizing).
     *
     * @param parent
     *            the parent connector
     * @throws IllegalStateException
     *             if a parent is given even though the connector already has a
     *             parent
     */
    public void setParent(HasComponents parent) {
        // In FW this method handled the attach/detach and made sure old parent
        // was marked dirty. Now it has no other meaning than to make sure old
        // parent is marked dirty when present. Everything else happens other
        // ways.
        final Component oldParent = getParent().orElse(null);
        if (parent != null && oldParent != null) {
            throw new IllegalStateException(
                    getClass().getName() + " already has a parent.");
        }
        if (Objects.equals(oldParent, parent)) {
            return;
        }
        if (oldParent instanceof AbstractComponent) {
            ((AbstractComponent) oldParent).markAsDirtyRecursive();
        }
    }

    /**
     * Emits the component event. It is transmitted to all registered listeners
     * interested in such events.
     */
    protected void fireComponentEvent() {
        fireEvent(new Event(this));
    }

    /**
     * Sets the focus for this component if the component is
     * {@link com.vaadin.flow.component.Focusable}. <em>NOTE:</em> the focus is
     * not set until the component is attached.
     */
    protected void focus() {
        // this is a NOOP here as any classic component that actually implements
        // Focusable, it will have focus() as public method
    }

    /**
     * Determine whether a <code>content</code> component is equal to, or the
     * ancestor of this component.
     *
     * @param content
     *            the potential ancestor element
     * @return <code>true</code> if the relationship holds
     */
    protected boolean isOrHasAncestor(Component content) {
        if (content instanceof com.vaadin.flow.component.HasComponents
                || content instanceof HasComponents) {
            if (Objects.equals(this, content)) {
                return true;
            }
            Optional<Component> optionalParent = getParent();
            while (optionalParent.isPresent()) {
                final Component parent = optionalParent.get();
                if (parent.equals(content)) {
                    return true;
                }
                optionalParent = parent.getParent();
            }
        }
        return false;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setWidth(java.lang.String)
     */
    @Override
    public void setWidth(String width) {
        final SizeWithUnit parsedWidth = SizeWithUnit.parseStringSize(width);
        // delegating to this method as it is overridden in subclasses
        if (parsedWidth != null) {
            setWidth(parsedWidth.getSize(), parsedWidth.getUnit());
        } else {
            setWidth(-1, Unit.PIXELS);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setWidth(float, Unit)
     */
    @Override
    public void setWidth(float width, Unit unit) {
        if (unit == null) {
            throw new IllegalArgumentException("Unit can not be null");
        }
        doSetWidth(new SizeWithUnit(width, unit));
    }

    @Override
    public float getWidth() {
        // 1-1 with FW behavior: returns the value set instead of what is sent
        // to client side
        return cachedWidth == null ? SIZE_UNDEFINED : cachedWidth.getSize();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#getWidthUnits()
     */
    @Override
    public Unit getWidthUnits() {
        return cachedWidth == null ? Unit.PIXELS : cachedWidth.getUnit();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setHeight(java.lang.String)
     */
    @Override
    public void setHeight(String height) {
        final SizeWithUnit parsedHeight = SizeWithUnit.parseStringSize(height);
        // delegating to this method as it is overridden in subclasses
        if (parsedHeight != null) {
            setHeight(parsedHeight.getSize(), parsedHeight.getUnit());
        } else {
            setHeight(-1, Unit.PIXELS);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setHeight(float, Unit)
     */
    @Override
    public void setHeight(float height, Unit unit) {
        if (unit == null) {
            throw new IllegalArgumentException("Unit can not be null");
        }
        doSetHeight(new SizeWithUnit(height, unit));
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.Sizeable#getHeight()
     */
    @Override
    public float getHeight() {
        // 1-1 with FW behavior: returns the value set instead of what is sent
        // to client side
        return cachedHeight == null ? SIZE_UNDEFINED : cachedHeight.getSize();
    }

    @Override
    public Unit getHeightUnits() {
        return cachedHeight == null ? Unit.PIXELS : cachedHeight.getUnit();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setSizeFull()
     */
    @Override
    public void setSizeFull() {
        setWidth(100, Unit.PERCENTAGE);
        setHeight(100, Unit.PERCENTAGE);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setWidthFull()
     */
    @Override
    public void setWidthFull() {
        setWidth(100, Unit.PERCENTAGE);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setHeightFull()
     */
    @Override
    public void setHeightFull() {
        setHeight(100, Unit.PERCENTAGE);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setSizeUndefined()
     */
    @Override
    public void setSizeUndefined() {
        setWidthUndefined();
        setHeightUndefined();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setWidthUndefined()
     */
    @Override
    public void setWidthUndefined() {
        setWidth(-1, Unit.PIXELS);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.vaadin.server.Sizeable#setHeightUndefined()
     */
    @Override
    public void setHeightUndefined() {
        setHeight(-1, Unit.PIXELS);
    }

    /**
     * Build CSS compatible string representation of height.
     *
     * @return CSS height
     */
    private String getCSSHeight() {
        return getHeight() + getHeightUnits().getSymbol();
    }

    /**
     * Build CSS compatible string representation of width.
     *
     * @return CSS width
     */
    private String getCSSWidth() {
        return getWidth() + getWidthUnits().getSymbol();
    }

    /**
     * Internal method for reacting to width changes.
     *
     * @param width
     *            the changed width to set
     */
    protected void doSetWidth(SizeWithUnit width) {
        cachedWidth = width;
        getParent().ifPresent(parent -> {
            if (parent instanceof AbstractComponentContainer) {
                // the size of siblings might have changed
                ((AbstractComponentContainer) parent).markAsDirtyRecursive();
            } else {
                markAsDirty();
            }
        });
    }

    /**
     * Internal method for reacting to height changes.
     *
     * @param height
     *            the changed height to set
     */
    protected void doSetHeight(SizeWithUnit height) {
        cachedHeight = height;
        getParent().ifPresent(parent -> {
            if (parent instanceof AbstractComponentContainer) {
                // the size of siblings might have changed
                ((AbstractComponentContainer) parent).markAsDirtyRecursive();
            } else {
                markAsDirty();
            }
        });
    }

    /**
     * Gets all user-defined CSS style names of a classic component. If the
     * component has multiple style names defined, the return string is a
     * space-separated list of style names. Built-in style names defined in
     * Vaadin or GWT are not returned.
     *
     * <p>
     * The style names are returned only in the basic form in which they were
     * added; each user-defined style name shows as two CSS style class names in
     * the rendered HTML: one as it was given and one prefixed with the
     * component-specific style name. Only the former is returned.
     * </p>
     *
     * @return the style name or a space-separated list of user-defined style
     *         names of the component
     * @see #setStyleName(String)
     * @see #addStyleName(String)
     * @see #removeStyleName(String)
     */
    public String getStyleName() {
        return customStyles == null ? ""
                : customStyles.stream().collect(Collectors.joining(" "));
    }

    /**
     * Sets one or more user-defined style names of the classic component,
     * replacing any previous user-defined styles. Multiple styles can be
     * specified as a space-separated list of style names. The style names must
     * be valid CSS class names and should not conflict with any built-in style
     * names in Vaadin or GWT.
     *
     * <pre>
     * Label label = new Label(&quot;This text has a lot of style&quot;);
     * label.setStyleName(&quot;myonestyle myotherstyle&quot;);
     * </pre>
     *
     * <p>
     * Each style name will occur in two versions: one as specified and one that
     * is prefixed with the style name of the component. For example, if you
     * have a {@code Label} component and give it "{@code mystyle}" style, the
     * component will have both "{@code mystyle}" and "{@code v-label-mystyle}"
     * styles. You could then style the component either with:
     * </p>
     *
     * <pre>
     * .mystyle {background: blue;}
     * </pre>
     *
     * <p>
     * or
     * </p>
     *
     * <pre>
     * .v-label-mystyle {background: blue;}
     * </pre>
     *
     * <p>
     * It is normally a good practice to use {@link #addStyleName(String)
     * addStyleName()} rather than this setter, as different software
     * abstraction layers can then add their own styles without accidentally
     * removing those defined in other layers.
     * </p>
     *
     * @param style
     *            the new style or styles of the component as a space-separated
     *            list
     * @see #getStyleName()
     * @see #addStyleName(String)
     * @see #removeStyleName(String)
     */
    public void setStyleName(String style) {
        if (style == null || style.isEmpty()) {
            customStyles = null;
            markAsDirty();
            return;
        }

        if (customStyles == null) {
            customStyles = new ArrayList<>();
        } else {
            customStyles.clear();
        }
        StringTokenizer tokenizer = new StringTokenizer(style, " ");
        while (tokenizer.hasMoreTokens()) {
            customStyles.add(tokenizer.nextToken());
        }
        markAsDirty();
        notifyParentAboutStyleChanges();
    }

    /**
     * Adds or removes a style name. Multiple styles can be specified as a
     * space-separated list of style names.
     * <p>
     * If the {@code add} parameter is true, the style name is added to the
     * component. If the {@code add} parameter is false, the style name is
     * removed from the component.
     * <p>
     * Functionally this is equivalent to using {@link #addStyleName(String)} or
     * {@link #removeStyleName(String)}
     *
     * @param style
     *            the style name to be added or removed
     * @param add
     *            <code>true</code> to add the given style, <code>false</code>
     *            to remove it
     * @see #addStyleName(String)
     * @see #removeStyleName(String)
     */
    public void setStyleName(String style, boolean add) {
        if (add) {
            addStyleName(style);
        } else {
            removeStyleName(style);
        }
    }

    /**
     * Adds one or more style names to the classic component. Multiple styles can
     * be specified as a space-separated list of style names. The style name
     * will be rendered as a HTML class name, which can be used in a CSS
     * definition.
     *
     * <pre>
     * Label label = new Label(&quot;This text has style&quot;);
     * label.addStyleName(&quot;mystyle&quot;);
     * </pre>
     *
     * <p>
     * Each style name will occur in two versions: one as specified and one that
     * is prefixed with the style name of the component. For example, if you
     * have a {@code Label} component and give it "{@code mystyle}" style, the
     * component will have both "{@code mystyle}" and "{@code v-label-mystyle}"
     * styles. You could then style the component either with:
     * </p>
     *
     * <pre>
     * .mystyle {font-style: italic;}
     * </pre>
     *
     * <p>
     * or
     * </p>
     *
     * <pre>
     * .v-label-mystyle {font-style: italic;}
     * </pre>
     *
     * @param style
     *            the new style to be added to the component
     * @see #getStyleName()
     * @see #setStyleName(String)
     * @see #removeStyleName(String)
     */
    public void addStyleName(String style) {
        if (style == null || style.isEmpty()) {
            return;
        }
        if (customStyles != null && customStyles.contains(style)) {
            return;
        }

        if (style.contains(" ")) {
            // Split space separated style names and add them one by one.
            StringTokenizer tokenizer = new StringTokenizer(style, " ");
            while (tokenizer.hasMoreTokens()) {
                addStyleName(tokenizer.nextToken());
            }
            return;
        }

        if (customStyles == null) {
            customStyles = new ArrayList<>();
        }
        customStyles.add(style);
        markAsDirty();
        notifyParentAboutStyleChanges();
    }

    /**
     * Adds one or more style names to this classic component by using one or
     * multiple parameters.
     *
     * @param styles
     *            the style name or style names to be added to the component
     * @see #addStyleName(String)
     * @see #setStyleName(String)
     * @see #removeStyleName(String)
     */
    public void addStyleNames(String... styles) {
        for (String style : styles) {
            addStyleName(style);
        }
    }

    /**
     * Removes one or more style names from the classic component. Multiple
     * styles can be specified as a space-separated list of style names.
     *
     * <p>
     * The parameter must be a valid CSS style name. Only user-defined style
     * names added with {@link #addStyleName(String) addStyleName()} or
     * {@link #setStyleName(String) setStyleName()} can be removed; built-in
     * style names defined in Vaadin or GWT can not be removed.
     * </p>
     *
     * @param style
     *            the style name or style names to be removed
     * @see #getStyleName()
     * @see #setStyleName(String)
     * @see #addStyleName(String)
     */
    public void removeStyleName(String style) {
        // the FW doesn't care for null checks ...
        StringTokenizer tokenizer = new StringTokenizer(style, " ");
        while (tokenizer.hasMoreTokens()) {
            customStyles.remove(tokenizer.nextToken());
        }
        markAsDirty();
        notifyParentAboutStyleChanges();
    }

    /**
     * Removes one or more style names from the classic component. Multiple
     * styles can be specified by using multiple parameters.
     *
     * @param styles
     *            the style name or style names to be removed
     * @see #removeStyleName(String)
     * @see #setStyleName(String)
     * @see #addStyleName(String)
     */
    public void removeStyleNames(String... styles) {
        for (String style : styles) {
            removeStyleName(style);
        }
    }

    /**
     * Gets the primary style name of the classic component. See
     * {@link #setPrimaryStyleName(String)} for a better description of the
     * primary stylename.
     */
    public String getPrimaryStyleName() {
        return primaryStyleName;
    }

    /**
     * Changes the primary style name of the classic component.
     *
     * <p>
     * The primary style name identifies the component when applying the CSS
     * theme to the Component. By changing the style name all CSS rules targeted
     * for that style name will no longer apply, and might result in the
     * component not working as intended.
     * </p>
     *
     * @param style
     *            The new primary style name
     */
    public void setPrimaryStyleName(String style) {
        if (!Objects.equals(style, primaryStyleName)) {
            primaryStyleName = style;
            markAsDirty();
        }
    }

    /**
     * This method is executed when a style name is added or removed, mainly
     * used to mark the parent as dirty. The default implementation marks the
     * parent as dirty if it's an AbstractOrderedLayout.
     */
    protected void notifyParentAboutStyleChanges() {
        // this ain't pretty, but need to update slot styles for HL+VL
        getParent().ifPresent(parent -> {
            if (parent instanceof AbstractOrderedLayout) {
                ((AbstractOrderedLayout) parent).markAsDirty();
            }
        });
        // when component is added/removed, part is marked dirty anyway
    }

    /**
     * Adds given style names as "internal" which means that those will not be
     * prefixed with the primary style name like user added style names are. In
     * legacy framework these were handled by the client side
     * connectors/widgets.
     *
     * @param internalStyles
     *            internal styles to add, not {@code null}
     */
    protected void addInternalStyles(String... internalStyles) {
        this.internalStyles.addAll(Arrays.asList(internalStyles));
        markAsDirty();
    }

    /**
     * Removes the given internal styles.
     *
     * @param internalStyles
     *            the internal styles to remove
     */
    protected void removeInternalStyles(String... internalStyles) {
        this.internalStyles.removeAll(Arrays.asList(internalStyles));
        markAsDirty();
    }

    /**
     * Returns the internal styles in an unmodifiable collection.
     *
     * @return the internal styles of the component
     *
     * @see #addInternalStyles(String...)
     * @see #removeInternalStyles(String...)
     */
    protected Collection<String> getInternalStyles() {
        if (internalStyles == null)
            return Collections.emptyList();
        else
            return Collections.unmodifiableList(internalStyles);
    }

    /**
     * Returns the user added custom style names in an unmodifiable collection.
     *
     * @return the user added custom style names
     */
    protected Collection<String> getCustomStyles() {
        if (customStyles == null)
            return Collections.emptyList();
        else
            return Collections.unmodifiableList(customStyles);
    }

    private String createClassName() {
        StringBuilder builder = new StringBuilder();
        final String primaryStyleName = getPrimaryStyleName();
        if (primaryStyleName != null) {
            builder.append(primaryStyleName);
        }
        if (!internalStyles.isEmpty()) {
            builder.append(" ").append(
                    internalStyles.stream().collect(Collectors.joining(" ")));
        }
        if (customStyles != null) {
            customStyles.forEach(style -> {
                if (primaryStyleName != null) {
                    builder.append(" ").append(primaryStyleName).append("-")
                            .append(style);
                }
                builder.append(" ").append(style);
            });
        }
        return builder.toString();
    }

    /**
     * Registers a new (generic) component event listener for the component.
     *
     * <pre>
     * class Listening extends CustomComponent implements Listener {
     *     // Stored for determining the source of an event
     *     Button ok;
     *
     *     Label status; // For displaying info about the event
     *
     *     public Listening() {
     *         VerticalLayout layout = new VerticalLayout();
     *
     *         // Some miscellaneous component
     *         TextField name = new TextField(&quot;Say it all here&quot;);
     *         name.addListener(this);
     *         layout.addComponent(name);
     *
     *         // Handle button clicks as generic events instead
     *         // of Button.ClickEvent events
     *         ok = new Button(&quot;OK&quot;);
     *         ok.addListener(this);
     *         layout.addComponent(ok);
     *
     *         // For displaying information about an event
     *         status = new Label(&quot;&quot;);
     *         layout.addComponent(status);
     *
     *         setCompositionRoot(layout);
     *     }
     *
     *     public void componentEvent(Event event) {
     *         // Act according to the source of the event
     *         if (event.getSource() == ok)
     *             getWindow().showNotification(&quot;Click!&quot;);
     *
     *         status.setValue(
     *                 &quot;Event from &quot; + event.getSource().getClass().getName()
     *                         + &quot;: &quot; + event.getClass().getName());
     *     }
     * }
     *
     * Listening listening = new Listening();
     * layout.addComponent(listening);
     * </pre>
     *
     * @param listener
     *            the new Listener to be registered.
     * @return a registration object for removing this listener
     * @see Event
     * @see Registration
     */
    public Registration addListener(Listener listener) {
        return addListener(Event.class, listener);
    }

    /*
     * When a component is disabled, FW just sets the `v-disabled` class for
     * most of the components. If a component needs to do something different it
     * has to override this method.
     */
    @Override
    public void onEnabledStateChanged(boolean enabled) {
        if (!enabled) {
            addInternalStyles("v-disabled");
        } else {
            removeInternalStyles("v-disabled");
        }
    }

    /**
     * Tests whether the component is enabled or not. A user can not interact
     * with disabled components. Disabled components are rendered in a style
     * that indicates the status, usually in gray color. Children of a disabled
     * component are also disabled. Components are enabled by default.
     *
     * <p>
     * As a security feature, all updates for disabled components are blocked on
     * the server-side.
     * </p>
     *
     * <p>
     * Note that this method only returns the status of the component and does
     * not take parents into account. Even though this method returns true the
     * component can be disabled to the user if a parent is disabled.
     * </p>
     *
     * @return <code>true</code> if the component and its parent are enabled,
     *         <code>false</code> otherwise.
     */
    @Override
    public boolean isEnabled() {
        return HasEnabled.super.isEnabled();
    }

    /**
     * Enables or disables the component. The user can not interact with
     * disabled components, which are shown with a style that indicates the
     * status, usually shaded in light gray color. Components are enabled by
     * default.
     *
     * <pre>
     * Button enabled = new Button(&quot;Enabled&quot;);
     * enabled.setEnabled(true); // The default
     * layout.addComponent(enabled);
     *
     * Button disabled = new Button(&quot;Disabled&quot;);
     * disabled.setEnabled(false);
     * layout.addComponent(disabled);
     * </pre>
     *
     * @param enabled
     *            a boolean value specifying if the component should be enabled
     *            or not
     */
    @Override
    public void setEnabled(boolean enabled) {
        HasEnabled.super.setEnabled(enabled);
    }

    /**
     * Sets the data object, that can be used for any application specific data.
     * The component does not use or modify this data.
     *
     * @param data
     *            the Application specific data.
     */
    public void setData(Object data) {
        ComponentUtil.setData(this, DataKey.class, new DataKey(data));
    }

    /**
     * Gets the application specific data. See {@link #setData(Object)}.
     *
     * @return the Application specific data set with setData function.
     */
    public Object getData() {
        DataKey data = ComponentUtil.getData(this, DataKey.class);
        if (data != null) {
            return data.getValue();
        }

        return null;
    }

    /**
     * Sets the locale of this component.
     *
     * <pre>
     * // Component for which the locale is meaningful
     * InlineDateField date = new InlineDateField(&quot;Datum&quot;);
     *
     * // German language specified with ISO 639-1 language
     * // code and ISO 3166-1 alpha-2 country code.
     * date.setLocale(new Locale(&quot;de&quot;, &quot;DE&quot;));
     *
     * date.setResolution(DateField.RESOLUTION_DAY);
     * layout.addComponent(date);
     * </pre>
     *
     * @param locale
     *            the locale to become this component's locale.
     */
    public void setLocale(Locale locale) {
        this.locale = locale;
        markAsDirty();
    }

    /**
     * Gets the locale of the component.
     *
     * <p>
     * If a component does not have a locale set, the locale of its parent is
     * returned, and so on. Eventually, if no parent has locale set, the locale
     * of the application is returned. If the application does not have a locale
     * set, it is determined by <code>Locale.getDefault()</code>.
     * </p>
     *
     * <p>
     * As the component must be attached before its locale can be acquired,
     * using this method in the internationalization of component captions, etc.
     * is generally not feasible. For such use case, we recommend using an
     * otherwise acquired reference to the application locale.
     * </p>
     *
     * @return Locale of this component or {@code null} if the component and
     *         none of its parents has a locale set and the component is not yet
     *         attached to an application.
     */
    @Override
    public Locale getLocale() {
        if (locale != null) {
            return locale;
        }

        Optional<AbstractComponent> parent = getFirstAbstractComponentParent(
                this);

        if (parent.isPresent()) {
            return parent.get().getLocale();
        }

        VaadinSession session = getSession();

        if (session != null) {
            return session.getLocale();
        }

        return null;
    }

    private Optional<AbstractComponent> getFirstAbstractComponentParent(
            Component component) {
        Optional<Component> parent = component.getParent();

        if (parent.isPresent() && parent.get() instanceof AbstractComponent) {
            return Optional.of((AbstractComponent) parent.get());
        } else if (parent.isPresent()) {
            return getFirstAbstractComponentParent(parent.get());
        }

        return Optional.empty();
    }

    /**
     * Gets the component visibility value.
     * <p>
     * Note: Although semantically the same, unlike Vaadin framework version 7/8
     * when a component is not visible, it is not removed from the DOM.
     *
     * @return {@code true} if the component is visible, {@code false} otherwise
     * @see #setVisible(boolean)
     */
    @Override
    public boolean isVisible() {
        return super.isVisible();
    }

    /**
     * Sets the component visibility value.
     * <p>
     * When a component is set as invisible, all the updates of the component
     * from the server to the client are blocked until the component is set as
     * visible again.
     * <p>
     * Invisible components don't receive any updates from the client-side.
     * Unlike the server-side updates, client-side updates, if any, are
     * discarded while the component is invisible, and are not transmitted to
     * the server when the component is made visible.
     * <p>
     * Note: Although semantically the same, unlike Vaadin framework version 7/8
     * when a component is not visible, it is not removed from the DOM.
     *
     * @param visible
     *            the component visibility value
     * @see #isVisible()
     */
    @Override
    public void setVisible(boolean visible) {
        super.setVisible(visible);
    }

    private static class DataKey {
        private final Object value;

        public DataKey(Object value) {
            this.value = value;
        }

        public Object getValue() {
            return value;
        }
    }

    /*
     * ----------------------------------------------------------------------
     * UNSUPPORTED API:
     * ----------------------------------------------------------------------
     */

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake registration
     */
    @Deprecated
    public Registration addShortcutListener(ShortcutListener shortcut) {
        final Class<? extends AbstractComponent> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass,
                "addShortcutListener(ShortcutListener)");
        return () -> Helpers.logUnsupportedApiCall(aClass,
                "Registration.remove() returned by " + aClass
                        + ".addShortcutListener(ShortcutListener)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake resource
     */
    @Deprecated
    public Resource getIcon() {
        final Class<? extends AbstractClientConnector> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass, "getIcon()");
        return new Resource() {
            @Override
            public String getMIMEType() {
                Helpers.logUnsupportedApiCall(aClass,
                        "Resource.getMIMEType() obtained via "
                                + aClass.getName() + ".getIcon()");
                return "";
            }
        };
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     */
    @Deprecated
    protected void fireComponentErrorEvent() {
        Helpers.logUnsupportedApiCall(getClass(), "fireComponentErrorEvent()");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake action manager
     */
    @Deprecated
    protected ActionManager getActionManager() {
        Helpers.logUnsupportedApiCall(getClass(), "getActionManager()");
        return new ActionManager();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return empty string
     */
    @Deprecated
    public String getCaption() {
        Helpers.logUnsupportedApiCall(getClass(), "getCaption()");
        return "";
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake error message
     */
    @Deprecated
    public ErrorMessage getComponentError() {
        final Class<? extends AbstractComponent> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass, "getComponentError");
        return new ErrorMessage() {
            @Override
            public ErrorLevel getErrorLevel() {
                Helpers.logUnsupportedApiCall(aClass,
                        "ErrorMessage.getErrorLevel() obtained via " + aClass
                                + ".getComponentError()");
                return ErrorLevel.INFO;
            }

            @Override
            public String getFormattedHtmlMessage() {
                Helpers.logUnsupportedApiCall(aClass,
                        "ErrorMessage.getFormattedHtmlMessage() obtained via "
                                + aClass + ".getComponentError()");
                return "";
            }
        };
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return empty collection
     */
    @Deprecated
    protected Collection<String> getCustomAttributes() {
        Helpers.logUnsupportedApiCall(getClass(), "getCustomAttributes()");
        return Collections.emptyList();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return empty string
     */
    @Deprecated
    public String getDescription() {
        Helpers.logUnsupportedApiCall(getClass(), "getDescription()");
        return "";
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake error message
     */
    @Deprecated
    public ErrorMessage getErrorMessage() {
        final Class<? extends AbstractComponent> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass, "getErrorMessage()");
        return new ErrorMessage() {
            @Override
            public ErrorLevel getErrorLevel() {
                Helpers.logUnsupportedApiCall(aClass,
                        "ErrorMessage.getErrorLevel() obtained via " + aClass
                                + ".getErrorMessage()");
                return ErrorLevel.INFO;
            }

            @Override
            public String getFormattedHtmlMessage() {
                Helpers.logUnsupportedApiCall(aClass,
                        "ErrorMessage.getFormattedHtmlMessage() obtained via "
                                + aClass + ".getErrorMessage()");
                return "";
            }
        };
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake shared state
     */
    @Deprecated
    protected AbstractComponentState getState() {
        Helpers.logUnsupportedApiCall(getClass(), "getState()");
        return new AbstractComponentState();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return fake shared state
     */
    @Deprecated
    protected AbstractComponentState getState(boolean markAsDirty) {
        Helpers.logUnsupportedApiCall(getClass(), "getState(boolean)");
        return new AbstractComponentState();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return false
     */
    @Deprecated
    public boolean isCaptionAsHtml() {
        Helpers.logUnsupportedApiCall(getClass(), "isCaptionAsHtml()");
        return false;
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return false
     */
    @Deprecated
    protected boolean isReadOnly() {
        Helpers.logUnsupportedApiCall(getClass(), "isReadOnly()");
        return false;
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return false
     */
    @Deprecated
    protected boolean isRequiredIndicatorVisible() {
        Helpers.logUnsupportedApiCall(getClass(),
                "isRequireIndicatorVisible()");
        return false;
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @return false
     */
    @Deprecated
    public boolean isResponsive() {
        Helpers.logUnsupportedApiCall(getClass(), "isResponsive()");
        return false;
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param caption
     *            caption
     */
    @Deprecated
    public void setCaption(String caption) {
        Helpers.logUnsupportedApiCall(getClass(), "setCaption(String)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param captionAsHtml
     *            caption as html
     */
    @Deprecated
    public void setCaptionAsHtml(boolean captionAsHtml) {
        Helpers.logUnsupportedApiCall(getClass(), "setCaptionAsHtml(boolean)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param componentError
     *            component error
     */
    @Deprecated
    public void setComponentError(ErrorMessage componentError) {
        Helpers.logUnsupportedApiCall(getClass(),
                "setComponentError(ErrorMessage)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param description
     *            description
     */
    @Deprecated
    public void setDescription(String description) {
        Helpers.logUnsupportedApiCall(getClass(), "setDescription(String)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param description
     *            description
     * @param mode
     *            mode
     */
    @Deprecated
    public void setDescription(String description, ContentMode mode) {
        Helpers.logUnsupportedApiCall(getClass(),
                "setDescription(String, ContentMode)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param icon
     *            icon
     */
    @Deprecated
    public void setIcon(Resource icon) {
        Helpers.logUnsupportedApiCall(getClass(), "setIcon(Resource)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param readOnly
     *            read only
     */
    @Deprecated
    protected void setReadOnly(boolean readOnly) {
        Helpers.logUnsupportedApiCall(getClass(), "setReadOnly(boolean)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param visible
     *            visible
     */
    @Deprecated
    protected void setRequiredIndicatorVisible(boolean visible) {
        Helpers.logUnsupportedApiCall(getClass(),
                "setRequiredIndicatorVisible(boolean)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @param responsive
     *            responsive
     */
    @Deprecated
    public void setResponsive(boolean responsive) {
        Helpers.logUnsupportedApiCall(getClass(), "setResponsive(boolean)");
    }
}
