/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import java.io.Serializable;
import java.util.Iterator;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.shared.Registration;

/**
 * Interface that must be implemented by all legacy framework
 * {@link AbstractComponent}s that contain other {@link AbstractComponent}s and
 * Flow's {@link Component}s.
 *
 * @author Vaadin Ltd
 *
 */
public interface HasComponents extends Iterable<Component> {

    /**
     * Gets an iterator to the collection of contained components. Using this
     * iterator it is possible to step through all components contained in this
     * container.
     * <p>
     * The iterator is typically unmodifiable, and calls to
     * {@link Iterator#remove()} throw an exception.
     *
     * @return the component iterator.
     */
    @Override
    public Iterator<Component> iterator();

    /**
     * Interface for {@link HasComponents} implementations that support sending
     * attach and detach events for components.
     */
    public interface ComponentAttachDetachNotifier extends Serializable {
        /**
         * Listens the component attach events.
         *
         * @see Registration
         *
         * @param listener
         *            the listener to add, not null
         * @return a registration object for removing the listener
         */
        public Registration addComponentAttachListener(
                ComponentAttachListener listener);

        /**
         * Listens the component detach events.
         */
        public Registration addComponentDetachListener(
                ComponentDetachListener listener);

    }

    /**
     * Component attach listener interface.
     */
    @FunctionalInterface
    public interface ComponentAttachListener
            extends ComponentEventListener<ComponentAttachEvent> {

        /**
         * A new component is attached to container.
         *
         * @param event
         *            the component attach event.
         */
        public void componentAttachedToContainer(ComponentAttachEvent event);

        @Override
        default void onComponentEvent(
                ComponentAttachEvent componentAttachEvent) {
            componentAttachedToContainer(componentAttachEvent);
        }
    }

    /**
     * Component detach listener interface.
     */
    @FunctionalInterface
    public interface ComponentDetachListener
            extends ComponentEventListener<ComponentDetachEvent> {

        /**
         * A component has been detached from container.
         *
         * @param event
         *            the component detach event.
         */
        public void componentDetachedFromContainer(ComponentDetachEvent event);

        @Override
        default void onComponentEvent(
                ComponentDetachEvent componentDetachEvent) {
            componentDetachedFromContainer(componentDetachEvent);
        }
    }

    /**
     * Component attach event sent when a component is attached to container.
     */
    @SuppressWarnings("serial")
    class ComponentAttachEvent extends Event {

        private final Component component;

        /**
         * Creates a new attach event.
         *
         * @param container
         *            the container the component has been detached to.
         * @param attachedComponent
         *            the component that has been attached.
         */
        public ComponentAttachEvent(Component container,
                Component attachedComponent) {
            super(container, false);
            component = attachedComponent;
        }

        /**
         * Gets the component container.
         *
         */
        public HasComponents getContainer() {
            return (HasComponents) getSource();
        }

        /**
         * Gets the attached component.
         *
         */
        public Component getAttachedComponent() {
            return component;
        }
    }

    /**
     * Component detach event sent when a component is detached from container.
     */
    @SuppressWarnings("serial")
    class ComponentDetachEvent extends Event {

        private final Component component;

        /**
         * Creates a new detach event.
         *
         * @param container
         *            the container the component has been detached from.
         * @param detachedComponent
         *            the component that has been detached.
         */
        public ComponentDetachEvent(Component container,
                Component detachedComponent) {
            super(container, false);
            component = detachedComponent;
        }

        /**
         * Gets the component container.
         *
         */
        public HasComponents getContainer() {
            return (HasComponents) getSource();
        }

        /**
         * Gets the detached component.
         *
         * @return the detached component.
         */
        public Component getDetachedComponent() {
            return component;
        }
    }

}
