/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.server;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEvent;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.function.SerializableEventListener;
import com.vaadin.flow.internal.ExecutionContext;
import com.vaadin.flow.internal.StateTree;
import com.vaadin.flow.internal.UsageStatistics;
import com.vaadin.flow.server.ErrorHandler;
import com.vaadin.flow.server.VaadinRequest;
import com.vaadin.flow.server.VaadinResponse;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.Registration;
import com.vaadin.classic.v8.shared.communication.ServerRpc;
import com.vaadin.classic.v8.shared.communication.SharedState;

import elemental.json.JsonValue;

/**
 * An abstract base class for legacy Vaadin framework version 7/8
 * ClientConnector implementations. This class maps the basic functionality
 * required for connectors to work inside newer Vaadin versions (with Flow).
 *
 * @author Vaadin Ltd
 */
public abstract class AbstractClientConnector extends Component
        implements ClientConnector {

    private final static String PROJECT_NAME = "vaadin-classic-components";
    private final static String PROJECT_VERSION_PROP = "classic-components.version";

    private StateTree.ExecutionRegistration beforeClientResponseRegistration;
    private boolean initialResponse = true;

    static {
        String projectVersion = Helpers.getProperty(PROJECT_VERSION_PROP);
        Helpers.checkLicense(PROJECT_NAME, projectVersion);
        // no need to check for prod-mod as flow does it
        UsageStatistics.markAsUsed(PROJECT_NAME, projectVersion);
    }

    @Override
    protected void onAttach(com.vaadin.flow.component.AttachEvent attachEvent) {
        super.onAttach(attachEvent);
        attach();
    }

    @Override
    protected void onDetach(com.vaadin.flow.component.DetachEvent detachEvent) {
        super.onDetach(detachEvent);
        detach();
        if (beforeClientResponseRegistration != null) {
            // unregistering so flow won't keep the task in queue for now
            beforeClientResponseRegistration.remove();
            beforeClientResponseRegistration = null;
        }
    }

    @Override
    public void beforeClientResponse(boolean initial) {
        // default does nothing
    }

    @Override
    public void markAsDirty() {
        // session is null in almost all the tests ...
        assert getSession() == null || getSession().hasLock();
        // not doing anything if not attached as attach() will trigger it anyway
        if (beforeClientResponseRegistration == null && isAttached()) {
            getUI().ifPresent(ui -> {
                beforeClientResponseRegistration = ui.beforeClientResponse(this,
                        this::internalBeforeClientResponse);
            });
        }
    }

    @Override
    public void markAsDirtyRecursive() {
        markAsDirty();

        getChildren().forEach(AbstractClientConnector::markAsDirtyRecursive);
    }

    private static void markAsDirtyRecursive(Component child) {
        // as the FW did recursive calls, so does this
        if (child instanceof AbstractClientConnector) {
            ((AbstractClientConnector) child).markAsDirtyRecursive();
        } else {
            child.getChildren()
                    .forEach(AbstractClientConnector::markAsDirtyRecursive);
        }
    }

    private void internalBeforeClientResponse(
            ExecutionContext executionContext) {
        // in case of @PreserveOnRefresh, initial will be reported wrong
        beforeClientResponse(initialResponse);
        initialResponse = false;
        // no need to unregister as it is thrown away by Flow after first
        // execution
        beforeClientResponseRegistration = null;
    }

    @Override
    public Registration addLegacyAttachListener(AttachListener listener) {
        return addListener(AttachEvent.class, listener);
    }

    @Override
    public Registration addLegacyDetachListener(DetachListener listener) {
        return addListener(DetachEvent.class, listener);
    }

    /**
     * Checks if the given {@link ComponentEvent} type is listened for this
     * component.
     *
     * @param eventType
     *            the event type to be checked
     * @return true if a listener is registered for the given event type
     */
    protected boolean hasListeners(Class<? extends ComponentEvent> eventType) {
        return hasListener(eventType);
    }

    @Override
    public void attach() {
        markAsDirty();
        fireEvent(new AttachEvent(this));
    }

    @Override
    public void detach() {
        fireEvent(new DetachEvent(this));
    }

    /**
     * Delegates to {@link Component#isAttached()}. {@inheritDoc}
     *
     * @return true if the component is attached to an active UI.
     */
    @Override
    public boolean isAttached() {
        return super.isAttached();
    }

    @Override
    public boolean isConnectorEnabled() {
        return isAttached() && isVisible() && getElement().isEnabled();
    }

    /**
     * Finds the {@link VaadinSession} to which this connector belongs. If the
     * connector has not been attached, <code>null</code> is returned.
     *
     * @return The connector's session, or <code>null</code> if not attached
     */
    protected VaadinSession getSession() {
        return getUI().map(UI::getSession).orElse(null);
    }

    /*
     * ----------------------------------------------------------------------
     * UNSUPPORTED API:
     * ----------------------------------------------------------------------
     */

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param extension
     *            extension
     */
    @Deprecated
    protected void addExtension(Extension extension) {
        Helpers.logUnsupportedApiCall(getClass(), "addExtension(Extension)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param eventType
     *            event type
     * @param listener
     *            listener
     * @param method
     *            method
     * @return registration that does nothing
     */
    @Deprecated
    public Registration addListener(Class<?> eventType,
            SerializableEventListener listener, Method method) {
        final Class<? extends AbstractClientConnector> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass,
                "addListener(Class<?>, SerializableEventListener, Method)");
        return () -> Helpers.logUnsupportedApiCall(aClass,
                "Registration.remove() returned by " + aClass.getName()
                        + ".addListener");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param eventIdentifier
     *            event identifier
     * @param eventType
     *            event type
     * @param listener
     *            listener
     * @param method
     *            method
     * @return a registration that does nothing
     */
    @Deprecated
    protected Registration addListener(String eventIdentifier,
            Class<?> eventType, SerializableEventListener listener,
            Method method) {
        final Class<? extends AbstractClientConnector> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass,
                "addListener(String, Class<?>, SerializableEventListener, Method)");
        return () -> Helpers.logUnsupportedApiCall(aClass,
                "Registration.remove() returned by " + aClass.getName()
                        + ".addListener");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param interfaceName
     *            interface name
     * @param method
     *            method
     * @param parameters
     *            parameters
     */
    @Deprecated
    protected void addMethodInvocationToQueue(String interfaceName,
            Method method, Object[] parameters) {
        Helpers.logUnsupportedApiCall(getClass(),
                "addMethodInvocationToQueue(String, Method, Object[])");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @return placeholder shared state
     */
    @Deprecated
    protected SharedState createState() {
        Helpers.logUnsupportedApiCall(getClass(), "createState()");
        return new SharedState();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param connector
     *            connector
     * @return empty iterator
     */
    @Deprecated
    public static Iterable<? extends ClientConnector> getAllChildrenIterable(
            ClientConnector connector) {
        Helpers.logUnsupportedApiCall(AbstractClientConnector.class,
                "getAllChildrenIterable(ClientConnector)");
        return (Iterable<? extends ClientConnector>) Collections
                .emptyIterator();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode*
     * @return empty string
     */
    @Deprecated
    public String getConnectorId() {
        Helpers.logUnsupportedApiCall(getClass(), "getConnectorId()");
        return "";
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @return error handler that does nothing
     */
    @Deprecated
    public ErrorHandler getErrorHandler() {
        final Class<? extends AbstractClientConnector> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass, "getErrorHandler()");
        return errorEvent -> {
            Helpers.logUnsupportedApiCall(aClass,
                    "ErrorHandler::onError(ErrorEvent) returned by "
                            + aClass.getName() + ".getErrorHandler()");
        };
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @return an empty collection
     */
    @Deprecated
    public Collection<Extension> getExtensions() {
        Helpers.logUnsupportedApiCall(getClass(), "getExtension()");
        return Collections.emptyList();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param key
     *            key
     * @return a fake resource object
     */
    @Deprecated
    protected Resource getResource(String key) {
        final Class<? extends AbstractClientConnector> aClass = getClass();
        Helpers.logUnsupportedApiCall(aClass, "getResource(String)");
        return new Resource() {
            @Override
            public String getMIMEType() {
                Helpers.logUnsupportedApiCall(aClass,
                        "Resource::getMIMEType() returned by "
                                + aClass.getName() + ".getResource()");
                return "";
            }
        };
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @return placeholder shared state
     */
    @Deprecated
    protected SharedState getState() {
        Helpers.logUnsupportedApiCall(getClass(), "getState()");
        return new SharedState();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param markAsDirty
     *            mark as dirty
     * @return placeholder shared state
     */
    @Deprecated
    protected SharedState getState(boolean markAsDirty) {
        Helpers.logUnsupportedApiCall(getClass(), "getState(boolean)");
        return new SharedState();
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param request
     *            request
     * @param response
     *            response
     * @param path
     *            path
     * @return false
     */
    @Deprecated
    public boolean handleConnectorRequest(VaadinRequest request,
            VaadinResponse response, String path) {
        Helpers.logUnsupportedApiCall(getClass(),
                "handleConnectorRequest(VaadinRequest, VaadinResponse, String");
        return false;
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param implementation
     *            implementation
     * @param <T>
     *            type
     */
    @Deprecated
    protected <T extends ServerRpc> void registerRpc(T implementation) {
        Helpers.logUnsupportedApiCall(getClass(), "registerRpc(T)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param implementation
     *            implementation
     * @param rpcInterfaceType
     *            rpc interface type
     * @param <T>
     *            type
     */
    @Deprecated
    protected <T extends ServerRpc> void registerRpc(T implementation,
            Class<T> rpcInterfaceType) {
        Helpers.logUnsupportedApiCall(getClass(), "registerRpc(T, Class<T>)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param extension
     *            extension
     */
    @Deprecated
    public void removeExtension(Extension extension) {
        Helpers.logUnsupportedApiCall(getClass(), "removeExtension(Extension)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param errorHandler
     *            error handler
     */
    @Deprecated
    public void setErrorHandler(ErrorHandler errorHandler) {
        Helpers.logUnsupportedApiCall(getClass(),
                "setErrorHandler(ErrorHandler)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param key
     *            key
     * @param resource
     *            resource
     */
    @Deprecated
    protected void setResource(String key, Resource resource) {
        Helpers.logUnsupportedApiCall(getClass(),
                "setResource(String, Resource)");
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/legacy-component-pack for
     * mitigation options.
     *
     * @deprecated not supported - calling this does nothing but logs a warning
     *             in development mode
     * @param propertyName
     *            property name
     * @param newValue
     *            new value
     */
    @Deprecated
    protected void updateDiffstate(String propertyName, JsonValue newValue) {
        Helpers.logUnsupportedApiCall(getClass(),
                "updateDiffstate(String, JsonValue)");
    }
}
