/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.server;

import java.io.IOException;
import java.io.Serializable;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.flow.server.VaadinService;
import com.vaadin.pro.licensechecker.LicenseChecker;

/**
 * Helper class for classic component pack internal things.
 */
public class Helpers implements Serializable {

    public static final String UNSUPPORTED_API_LOGGER_NAME = "Unsupported API";
    public static final String VAADIN_7_SYSTEM_PROPERTY = "vaadin.classic-components.vaadin7defaults";
    private static final String VAADIN_7_DEFAULTS = System
            .getProperty(VAADIN_7_SYSTEM_PROPERTY);

    /**
     * A custom security manager that exposes the getClassContext() information
     */
    private static class CustomSecurityManager extends SecurityManager {

        /**
         * Gets the first non legacy pack class in the caller stack.
         *
         * @return the class name or null if none exists.
         */
        public String getFirstNonLegacyClassInCallStack() {
            final Class<?>[] classContext = getClassContext();
            for (int i = 2; i < classContext.length; i++) {
                final String name = classContext[i].getName();
                if (!name.contains("com.vaadin.classic")) {
                    return name;
                }
            }
            return null;
        }
    }

    private final static CustomSecurityManager CUSTOM_SECURITY_MANAGER = new CustomSecurityManager();

    private Helpers() {
        // Helpers methods only
    }

    static Properties getProperties(VaadinService service) {
        Properties properties = new Properties();
        try {
            properties.load(service.getClassLoader().getResourceAsStream(
                    "vaadin-classic-components-version.properties"));
        } catch (IOException e) {
            e.printStackTrace();
        }
        return properties;
    }

    static boolean isProductionMode(VaadinService service) {
        return service.getDeploymentConfiguration().isProductionMode();
    }

    static VaadinService getService() {
        return VaadinService.getCurrent();
    }

    static void checkLicense(String projectName, String version) {
        VaadinService service = getService();

        if (service != null) {
            if (!isProductionMode(service)) {
                LicenseChecker.checkLicenseFromStaticBlock(projectName,
                        version);
            }
        }
    }

    static String getProperty(String property) {
        VaadinService service = getService();
        if (service == null) {
            return null;
        }
        Properties properties = getProperties(service);
        return properties.getProperty(property);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UNSUPPORTED_API_LOGGER_NAME);
    }

    /**
     * Logs out a warning in development mode about unsupported API call of the
     * given method in the given class.
     *
     * @param clazz
     *            the class that was called
     * @param methodName
     *            the name of the unsupported method that was called
     */
    public static void logUnsupportedApiCall(Class<?> clazz,
            String methodName) {
        boolean productionMode = true;
        String callerClassName = null;
        String calledClassName = null;
        try {
            productionMode = isProductionMode(getService());
            if (!productionMode) {
                callerClassName = CUSTOM_SECURITY_MANAGER
                        .getFirstNonLegacyClassInCallStack();
                calledClassName = clazz.getName();
            }
        } catch (Exception e) {
            // something went wrong, cannot log calling class
        }
        if (!productionMode) {
            StringBuilder message = new StringBuilder("Calling: '");
            if (calledClassName != null) {
                message.append(calledClassName).append(".");
            }
            message.append(methodName);
            if (callerClassName != null) {
                message.append("'\nfrom app class: '").append(callerClassName);
            }
            message.append("'. This call has no effect.");
            getLogger().warn(message.toString());
        }
    }

    /**
     * Checks whether the Vaadin 7 defaults should be used or not.
     *
     * @return {@code true} to use V7 defaults, {@code false} to not and use V8
     *         defaults
     */
    public static boolean isVaadin7Defaults() {
        return VAADIN_7_DEFAULTS != null;
    }
}
