/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import com.vaadin.flow.component.Component;

/**
 * Legacy version of ComponentContainer, a special type of parent which allows
 * the user to add and remove components to it. Typically, does not have any
 * restrictions on the number of children it can contain.
 *
 * @author Vaadin Ltd.
 */
public interface ComponentContainer
        extends HasComponents, com.vaadin.flow.component.HasComponents,
        HasComponents.ComponentAttachDetachNotifier {
    /**
     * Adds the component into this container.
     *
     * @param c
     *            the component to be added.
     */
    void addComponent(Component c);

    /**
     * Adds the components in the given order to this component container.
     *
     * @param components
     *            The components to add.
     */
    void addComponents(Component... components);

    /**
     * Removes the component from this container.
     *
     * @param c
     *            the component to be removed.
     */
    void removeComponent(Component c);

    /**
     * Removes all components from this container.
     */
    void removeAllComponents();

    /**
     * Replaces the component in the container with another one without changing
     * position.
     *
     * <p>
     * This method replaces component with another one is such way that the new
     * component overtakes the position of the old component. If the old
     * component is not in the container, the new component is added to the
     * container. If the both component are already in the container, their
     * positions are swapped. Component attach and detach events should be taken
     * care as with add and remove.
     * </p>
     *
     * @param oldComponent
     *            the old component that will be replaced.
     * @param newComponent
     *            the new component to be replaced.
     */
    void replaceComponent(Component oldComponent, Component newComponent);

    /**
     * Gets the number of children this {@link ComponentContainer} has
     *
     * @return The number of child components this container has.
     */
    int getComponentCount();

    /**
     * Moves all components from another container into this container. The
     * components are removed from <code>source</code>.
     *
     * @param source
     *            the container which contains the components that are to be
     *            moved to this container.
     */
    void moveComponentsFrom(ComponentContainer source);
}
