/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEvent;

/**
 * Legacy version of Event class that resembles Vaadin 7/8's
 * <code>Component.Event</code> API as closely as possible in order to
 * facilitate migration to newer versions of Vaadin.
 *
 * Superclass of all component originated events.
 *
 * <p>
 * Events are the basis of all user interaction handling in Vaadin. To handle
 * events, you provide a listener object that receives the events of the
 * particular event type.
 * </p>
 *
 * <pre>
 * Button button = new Button(&quot;Click Me!&quot;);
 * button.addListener(new Button.ClickListener() {
 *     public void buttonClick(ClickEvent event) {
 *         getWindow().showNotification(&quot;Thank You!&quot;);
 *     }
 * });
 * layout.addComponent(button);
 * </pre>
 *
 * <p>
 * Notice that while each of the event types have their corresponding listener
 * types; the listener interfaces are not required to inherit the
 * {@code com.vaadin.legacy.ui.Listener} interface.
 * </p>
 *
 * @see Listener
 */
public class Event extends ComponentEvent<Component> {
    /**
     * Constructs a new event with the specified source component.
     *
     * @param source
     *            the source component of the event
     */
    public Event(Component source) {
        super(source, false);
    }

    /**
     * Creates a new event using the given source and indicator whether the
     * event originated from the client side or the server side.
     *
     * @param source
     *            the source component
     * @param fromClient
     *            <code>true</code> if the event originated from the client
     *            side, <code>false</code> otherwise
     */
    public Event(Component source, boolean fromClient) {
        super(source, fromClient);
    }

    /**
     * Gets the component where the event occurred.
     *
     * @return the source component of the event
     */
    public Component getComponent() {
        return getSource();
    }
}
