/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.ui;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasTheme;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.dom.Element;

/**
 * <p>
 *  Legacy version of FormLayout that resembles Vaadin 7/8's FormLayout
 *  API as closely as possible in order to facilitate migration to newer versions
 *  of Vaadin.
 * </p>
 * <p>
 * FormLayout is used to layout fields. FormLayout is a close relative of
 * {@link VerticalLayout}, but in FormLayout captions are rendered to the left of
 * their respective components.
 * </p>
 */
@Tag(Tag.DIV)
@CssImport("./ordered-layout.css")
@CssImport("./margin.css")
@JsModule("./form-layout-styles.js")
public class FormLayout extends AbstractOrderedLayout {

    private static final String CLASSNAME = "v-formlayout";
    public static final String IN_FORM_LAYOUT_THEME = "in-form-layout";

    private final Table table;

    /**
     * Creates a new a FormLayout.
     */
    public FormLayout() {
        super(true);
        table = new Table();
        this.add(table);
        setPrimaryStyleName(CLASSNAME);
        setMargin(new MarginInfo(true, false));
        addStyleName("outlined");
    }

    /**
     * Constructs a FormLayout and adds the given components to it.
     *
     * @param children Components to add to the FormLayout
     */
    public FormLayout(Component... children) {
        this();
        addComponents(children);
    }

    /**
     * This method currently has no effect as expand ratios are not implemented
     * in FormLayout.
     *
     * @param component which expand ratios is requested
     * @return expand ratio of given component, 0.0f by default.
     */
    @Override
    @Deprecated
    public float getExpandRatio(Component component) {
        return 0.0f;
    }

    /**
     * This method currently has no effect as expand ratios are not implemented
     * in FormLayout.
     *
     * @param component
     *            the component in this layout which expand ratio is to be set
     * @param ratio
     *            new expand ratio (greater or equal to 0)
     */
    @Override
    public void setExpandRatio(Component component, float ratio) {
        // do nothing
    }

    @Override
    public void addComponent(Component component) {
        if (component == this.table) {
            super.addComponent(component);
        } else {
            if (component instanceof HasTheme) {
                ((HasTheme) component).addThemeName(IN_FORM_LAYOUT_THEME);
            }
            FormRow row = new FormRow(component);
            this.table.addComponent(row);
        }
    }

    @Override
    public void removeComponent(Component component) {
        // iterate the table rows to get the one with the desired component
        // and if found, remove the row from the table
        this.table.getChildren()
                .map(row -> (FormRow) row)
                .filter(row -> row.getContent().equals(component))
                .findFirst()
                .ifPresent(this.table::remove);
    }

    @Override
    public int getComponentIndex(Component component) {
        return this.table.getChildren()
                .map(row -> (FormRow) row)
                .map(FormRow::getContent)
                .collect(Collectors.toList())
                .indexOf(component);
    }

    @Override
    public Component getComponent(int index) throws IndexOutOfBoundsException {
        if (index < 0 || index >= this.table.getComponentCount()) {
            throw new IndexOutOfBoundsException();
        }
        return this.table.getChildren()
                .skip(index)
                .findFirst()
                .map(Component::getElement)
                .map(rowElement -> rowElement.getChild(1))
                .map(contentCellElement -> contentCellElement.getChild(0))
                .flatMap(Element::getComponent)
                .orElseThrow();
    }

    @Override
    public void addComponent(Component component, int index) throws IndexOutOfBoundsException {
        int componentCount = table.getComponentCount();
        if (index < 0 || index > componentCount) {
            throw new IndexOutOfBoundsException();
        }
        if (componentCount == 0 || index == componentCount) {
            this.add(component);
            return;
        }
        if (component instanceof HasTheme) {
            ((HasTheme) component).addThemeName(IN_FORM_LAYOUT_THEME);
        }
        FormRow row = new FormRow(component);
        this.table.addComponentAtIndex(index, row);
    }

    Table getTable() {
        return this.table;
    }

    /**
     * This method currently has no effect since the FormLayout has the table as
     * its only child.
     */
    @Override
    protected void setSlotClasses() {
        // do nothing
    }

    @Override
    public Stream<Component> getChildren() {
        return this.table.getChildren()
                .map(row -> (FormRow) row)
                .map(FormRow::getContent);
    }

    /**
     * <p>
     * A row for a {@link FormLayout} with two cells (columns), one for the
     * caption and another one for the content.
     * </p>
     *
     * @see FormLayout
     * @see TableRow
     */
    private static final class FormRow extends TableRow {
        private final TableDataCell captionCell;
        private final TableDataCell contentCell;

        FormRow(Component component) {
            super();
            this.addStyleName("v-formlayout-row");
            this.captionCell = new TableDataCell();
            // set the content of the label cell using javascript
            component.getElement().executeJs(
                    // add asterisk to required fields
                    "if (this.hasAttribute('required')) { this.closest('tr').firstChild.innerHTML = `<span>${this.label}</span><span class=\"v-required-field-indicator\">*</span>` }" +
                    " else { this.closest('tr').firstChild.innerHTML = `<span>${this.label}</span>`; }" +
                    // clear label columns for checkboxes
                    "if (this.tagName.toLowerCase() === 'vaadin-checkbox') { this.closest('tr').firstChild.innerText = ''; }");
            this.captionCell.addStyleName("v-formlayout-captioncell");

            this.contentCell = new TableDataCell(component);
            this.contentCell.addStyleName("v-formlayout-contentcell");
            this.add(this.captionCell, this.contentCell);
        }

        /**
         * Get a reference to the component shown in the content cell.
         *
         * @return the Component shown in the content cell.
         */
        Component getContent() {
            return contentCell.getContent();
        }
    }
}
