/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.ui;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Objects;
import java.util.stream.Stream;

import com.vaadin.classic.v8.event.LayoutEvents;
import com.vaadin.classic.v8.event.LayoutEvents.LayoutClickEvent;
import com.vaadin.classic.v8.event.LayoutEvents.LayoutClickListener;
import com.vaadin.classic.v8.server.Helpers;
import com.vaadin.classic.v8.shared.ui.gridlayout.GridLayoutState.ChildComponentData;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.shared.Registration;

import elemental.json.Json;
import elemental.json.JsonArray;
import static com.vaadin.classic.v8.shared.ui.CssClassName.UI_LAYOUT;
import static com.vaadin.classic.v8.shared.ui.CssClassName.WIDGET;

/**
 * Classic Components version of V8's GridLayout.
 * <p>
 * A layout where the components are laid out on a grid using cell coordinates.
 *
 * <p>
 * The GridLayout also maintains a cursor for adding components in
 * left-to-right, top-to-bottom order.
 * </p>
 *
 * <p>
 * Each component in a <code>GridLayout</code> uses a defined
 * {@link Area area} (column1,row1,column2,row2) from the grid. The
 * components may not overlap with the existing components - if you try to do so
 * you will get an {@link OverlapsException}. Adding a component with cursor
 * automatically extends the grid by increasing the grid height.
 * </p>
 *
 * <p>
 * The grid coordinates, which are specified by a row and column index, always
 * start from 0 for the topmost row and the leftmost column.
 * </p>
 * <p>
 * Example coordinates structure of a 3x3 GridLayout (column, row):
 *
 * <table>
 *     <tr>
 *         <td>(0,0)</td>
 *         <td>(1,0)</td>
 *         <td>(2,0)</td>
 *     </tr>
 *     <tr>
 *         <td>(0,1)</td>
 *         <td>(1,1)</td>
 *         <td>(2,1)</td>
 *     </tr>
 *     <tr>
 *         <td>(0,2)</td>
 *         <td>(1,2)</td>
 *         <td>(2,2)</td>
 *     </tr>
 * </table>
 */
@Tag(Tag.DIV)
@JsModule("./grid-layout.js")
@CssImport("./grid-layout.css")
public class GridLayout extends AbstractLayout
        implements Layout.SpacingHandler, Layout.AlignmentHandler,
        Layout.MarginHandler, LayoutEvents.LayoutClickNotifier {

    public static final String PRIMARY_STYLE_NAME = "v-gridlayout";

    private int columns, rows;
    private Cell[][] cells;
    private final HashMap<Component, Cell> componentCellMap = new HashMap<>();
    private final LinkedList<Component> components = new LinkedList<>();
    private int cursorX, cursorY;
    private float[] colExpandRatios;
    private float[] rowExpandRatios;
    private boolean spacing = false;
    private Alignment defaultAlignment = Alignment.TOP_LEFT;
    private boolean hideEmptyRowsAndColumns = false;
    private MarginInfo marginInfo = new MarginInfo(false);

    /**
     * Constructs an empty (1x1) grid layout that is extended as needed.
     */
    public GridLayout() {
        this(1, 1);
    }

    /**
     * Constructor for a grid of given size (number of columns and rows).
     * <p>
     * The grid may grow or shrink later. Grid grows automatically if you add
     * components outside its area.
     *
     * @param columns Number of columns in the grid.
     * @param rows    Number of rows in the grid.
     */
    public GridLayout(int columns, int rows) {
        super();
        setPrimaryStyleName(PRIMARY_STYLE_NAME);
        addInternalStyles(UI_LAYOUT);
        addInternalStyles(WIDGET);
        this.cells = new Cell[rows][columns];
        this.rowExpandRatios = new float[rows];
        this.colExpandRatios = new float[columns];
        Arrays.fill(rowExpandRatios, 0);
        Arrays.fill(colExpandRatios, 0);
        this.setColumns(columns);
        this.setRows(rows);
        this.getElement().executeJs("window.Vaadin.ClassicComponents.gridLayoutConnector.initLazy(this)");
    }

    /**
     * Constructs a GridLayout of given size (number of columns and rows) and
     * adds the given components in order to the grid.
     *
     * @param columns  Number of columns in the grid.
     * @param rows     Number of rows in the grid.
     * @param children Components to add to the grid.
     * @see #addComponents(Component...)
     */
    public GridLayout(int columns, int rows, Component... children) {
        this(columns, rows);
        this.addComponents(children);
    }

    @Override
    public void beforeClientResponse(boolean initial) {
        super.beforeClientResponse(initial);
        if (!initial) {
            this.getElement().executeJs("window.Vaadin.ClassicComponents.gridLayoutConnector.initLazy(this)");
        }
    }

    @Override
    protected void onDetach(com.vaadin.flow.component.DetachEvent detachEvent) {
        super.onDetach(detachEvent);
    }

    /**
     * Adds the component into this container to the cursor position. If the
     * cursor position is already occupied, the cursor is moved forwards to find
     * free position. If the cursor goes out from the bottom of the grid, the
     * grid is automatically extended.
     *
     * @param component the component to be added, not <code>null</code>.
     */
    @Override
    public void addComponent(Component component) {
        Objects.requireNonNull(component, "Component must not be null");

        Cell availableArea;
        boolean done = false;
        do {
            availableArea = new Cell(null, cursorX, cursorY, cursorX, cursorY);
            try {
                checkExistingOverlaps(availableArea);
                done = true;
            } catch (OverlapsException e) {
                // add a new row when reaching the end of the grid
                if (cursorX == this.columns - 1 && cursorY == this.rows - 1) {
                    this.appendRow();
                }
                // move the cursor to the next cell
                space();
            }
        } while (!done);
        this.addComponent(component, cursorX, cursorY);
    }

    /**
     * Adds the component to the grid in cells column1,row1 (NortWest corner of
     * the area.) End coordinates (SouthEast corner of the area) are the same as
     * column1,row1. The coordinates are zero-based. Component width and height
     * is 1.
     *
     * @param component the component to be added, not <code>null</code>.
     * @param column    the column index, starting from 0.
     * @param row       the row index, starting from 0.
     * @throws OverlapsException    if the new component overlaps with any of
     *                              the components already in the grid.
     * @throws OutOfBoundsException if the cell is outside the grid area.
     */
    public void addComponent(Component component, int column, int row)
            throws OverlapsException, OutOfBoundsException {
        this.addComponent(component, column, row, column, row);
    }

    /**
     * <p>
     * Adds a component to the grid in the specified area. The area is defined
     * by specifying the upper left corner (column1, row1) and the lower right
     * corner (column2, row2) of the area. The coordinates are zero-based.
     * </p>
     *
     * <p>
     * If the area overlaps with any of the existing components already present
     * in the grid, the operation will fail and an {@link OverlapsException} is
     * thrown.
     * </p>
     *
     * @param component the component to be added, not <code>null</code>.
     * @param column1   the column of the upper left corner of the area <code>component</code>
     *                  is supposed to occupy. The leftmost column has index 0.
     * @param row1      the row of the upper left corner of the area <code>component</code> is
     *                  supposed to occupy. The topmost row has index 0.
     * @param column2   the column of the lower right corner of the area
     *                  <code>ccomponent</code> is supposed to occupy.
     * @param row2      the row of the lower right corner of the area <code>component</code>
     *                  is supposed to occupy.
     * @throws OverlapsException    if the new component overlaps with any of the components
     *                              already in the grid.
     * @throws OutOfBoundsException if the cells are outside the grid area.
     */
    public void addComponent(Component component, int column1, int row1,
                             int column2, int row2)
            throws OverlapsException, OutOfBoundsException {
        Objects.requireNonNull(component, "Component must not be null");

        // check the component does not already exist in container
        if (components.contains(component)) {
            throw new IllegalArgumentException("Component is already in container");
        }

        // create the area
        final Cell cell = new Cell(component, column1, row1, column2, row2);
        cell.setAlignment(defaultAlignment);
        // Checks the validity of the coordinates
        if (column2 < column1 || row2 < row1) {
            throw new IllegalArgumentException(String.format(
                    "Illegal coordinates for the component: %s!<=%s, %s!<=%s",
                    column1, column2, row1, row2));
        }
        if (column1 < 0 || row1 < 0 || column2 >= this.columns
                || row2 >= this.rows) {
            throw new OutOfBoundsException(cell);
        }
        // Checks that newItem does not overlap with existing items
        checkExistingOverlaps(cell);

        // Figure out where the component should be inserted in the list
        // Respect top-down, left-right ordering
        boolean added = false;
        int index = 0;
        for (Component c : components) {
            Cell existingArea = this.componentCellMap.get(c);
            if ((existingArea.y1 >= row1 && existingArea.x1 > column1)
                    || existingArea.y1 > row1) {
                components.add(index, component);
                added = true;
                break;
            }
            index++;
        }
        // add at last place if no other place was found
        if (!added) {
            components.addLast(component);
        }
        componentCellMap.put(component, cell);

        // Attempt to add with super call
        try {
            cell.getElement().setAttribute("x", String.valueOf(column1));
            cell.getElement().setAttribute("y", String.valueOf(row1));
            super.addComponentAtIndex(index, cell);
            this.cells[column1][row1] = cell;
        } catch (IllegalArgumentException e) {
            componentCellMap.remove(component);
            components.remove(component);
            throw e;
        }

        // update cursor position if it's within this area; use first position
        // outside this area, even if it's occupied
        if (cursorX >= column1 && cursorX <= column2
                && cursorY >= row1 && cursorY <= row2) {
            // cursor within area
            cursorX = column2 + 1; // one right of area
            if (cursorX >= this.columns) {
                // cursor overflows columns
                if (row2 + 1 == this.rows) {
                    // no more rows available, so place cursor at last cell
                    cursorX = column2;
                    cursorY = row2;
                } else {
                    // place cursor at first column
                    cursorX = 0;
                    // if the area spans the first column,
                    // move cursor down below the area,
                    // otherwise move it just one row down
                    cursorY = (column1 == 0 ? row2 : row1) + 1;
                }
            } else {
                // cursor does not overflow columns
                cursorY = row1;
            }
        }
        this.getElement().executeJs("setTimeout(() => this.$connector.update());");
    }

    @Override
    public void removeComponent(Component component) {
        if (component == null || !components.contains(component)) {
            return;
        }
        Cell cell = componentCellMap.remove(component);
        components.remove(component);
        super.removeComponent(cell);
    }

    /**
     * Removes the component specified by its cell coordinates.
     *
     * @param column the component's column.
     * @param row    the component's row.
     */
    public void removeComponent(int column, int row) {
        componentCellMap.values()
                .stream()
                .filter(cell -> cell.x1 == column && cell.y1 == row)
                .findFirst()
                .map(Cell::getContent)
                .ifPresent(this::removeComponent);
    }

    @Override
    public void removeAllComponents() {
        super.removeAllComponents();
        this.setCursorX(0);
        this.setCursorY(0);
    }

    /**
     * Gets the Component at given index.
     *
     * @param x The column index, starting from 0 for the leftmost column.
     * @param y The row index, starting from 0 for the topmost row.
     * @return Component in given cell or null if empty
     */
    public Component getComponent(int x, int y) {
        return componentCellMap.values().stream()
                .filter(cell -> cell.x1 <= x && x <= cell.x2
                        && cell.y1 <= y && y <= cell.y2)
                .findFirst()
                .map(Cell::getContent)
                // We can't just return an Optional<Component> because we need
                // to mimic V8's API.
                .orElse(null);
    }

    /**
     * Returns information about the area where given component is laid in the
     * GridLayout.
     *
     * @param component the component whose area information is requested.
     * @return an {@link Area} object that contains information how the
     * component is laid in the grid.
     */
    public Area getComponentArea(Component component) {
        Cell cell = componentCellMap.get(component);
        if (cell == null) {
            return null;
        } else {
            return new Area(component, cell.x1, cell.y1, cell.x2, cell.y2);
        }
    }

    /**
     * Not supported. See Classic Component Pack documentation in
     * https://vaadin.com/docs/latest/flow/upgrading/classic-component-pack for
     * mitigation options.
     *
     * @deprecated
     */
    @Override
    public Registration addLayoutClickListener(LayoutClickListener listener) {
        return addListener("lClick", LayoutClickEvent.class, null, null);
    }

    @Deprecated
    public void removeLayoutClickListener(LayoutClickListener listener) {
        Helpers.logUnsupportedApiCall(getClass(), "removeLayoutClickListener");
    }

    @Override
    public void setComponentAlignment(Component childComponent, Alignment alignment) {
        Cell cell = componentCellMap.get(childComponent);
        if (cell != null) {
            cell.setAlignment(alignment);
            this.getElement().executeJs("setTimeout(() => this.$connector.update());");
        }
    }

    @Override
    public Alignment getComponentAlignment(Component childComponent) {
        Cell cell = componentCellMap.get(childComponent);
        return cell == null ? null : cell.getAlignment();
    }

    @Override
    public void setDefaultComponentAlignment(Alignment defaultComponentAlignment) {
        this.defaultAlignment = defaultComponentAlignment;
    }

    @Override
    public Alignment getDefaultComponentAlignment() {
        return defaultAlignment;
    }

    @Override
    public void setMargin(boolean enabled) {
        setMargin(new MarginInfo(enabled));
    }

    @Override
    public void setMargin(MarginInfo marginInfo) {
        this.marginInfo = marginInfo;
        if (marginInfo.hasTop()) {
            this.addStyleName("margin-top");
        } else {
            this.removeStyleName("margin-top");
        }
        if (marginInfo.hasRight()) {
            this.addStyleName("margin-right");
        } else {
            this.removeStyleName("margin-right");
        }
        if (marginInfo.hasBottom()) {
            this.addStyleName("margin-bottom");
        } else {
            this.removeStyleName("margin-bottom");
        }
        if (marginInfo.hasLeft()) {
            this.addStyleName("margin-left");
        } else {
            this.removeStyleName("margin-left");
        }
        this.getElement().executeJs("this.$connector.update();");
    }

    @Override
    public MarginInfo getMargin() {
        return this.marginInfo;
    }

    /**
     * Sets whether empty rows and columns should be considered as non-existent
     * when rendering or not. If this is set to true then the spacing between
     * multiple empty columns (or rows) will be collapsed.
     * <p>
     * The default behavior is to consider all rows and columns as visible
     * <p>
     * NOTE that this must be set before the initial rendering takes place.
     * Updating this on the fly is not supported.
     *
     * @param hideEmptyRowsAndColumns true to hide empty rows and columns, false to leave them as-is
     */
    public void setHideEmptyRowsAndColumns(boolean hideEmptyRowsAndColumns) {
        this.hideEmptyRowsAndColumns = hideEmptyRowsAndColumns;
        this.getElement().executeJs("setTimeout(() => { this.$connector.hideEmptyRowsAndColumns = "
                + hideEmptyRowsAndColumns + "; this.$connector.update(); })");
    }

    /**
     * Checks whether empty rows and columns should be considered as
     * non-existent when rendering or not.
     *
     * @return true if empty rows and columns are hidden, false otherwise
     * @see #setHideEmptyRowsAndColumns(boolean)
     */
    public boolean isHideEmptyRowsAndColumns() {
        return this.hideEmptyRowsAndColumns;
    }

    /**
     * Sets the expand-ratio of given row.
     *
     * <p>
     * Expand ratio defines how excess space is distributed among rows. Excess
     * space means the space left over from components that are not sized
     * relatively. By default, the excess space is distributed evenly.
     * </p>
     *
     * <p>
     * Note, that height of this GridLayout needs to be defined (fixed or
     * relative, as opposed to undefined height) for this method to have any
     * effect.
     * <p>
     * Note that checking for relative height for the child components is done
     * on the server so you cannot set a child component to have undefined
     * height on the server and set it to <code>100%</code> in CSS. You must set
     * it to <code>100%</code> on the server.
     *
     * @param rowIndex The row index, starting from 0 for the topmost row.
     * @param ratio    the expand-ratio
     * @see #setHeight(float, Unit)
     */
    public void setRowExpandRatio(int rowIndex, float ratio) {
        this.rowExpandRatios[rowIndex] = ratio;
        // update client property
        JsonArray rowExpandRatios = Json.createArray();
        for (int i = 0; i < this.rowExpandRatios.length; i++) {
            rowExpandRatios.set(i, this.rowExpandRatios[i]);
        }
        this.getElement().setPropertyJson("rowExpandRatios", rowExpandRatios);
        this.getElement().executeJs("setTimeout(() => this.$connector.update());");
        markAsDirty();
    }

    /**
     * Returns the expand-ratio of given row.
     *
     * @param rowIndex The row index, starting from 0 for the topmost row.
     * @return the expand ratio, 0.0f by default
     * @see #setRowExpandRatio(int, float)
     */
    public float getRowExpandRatio(int rowIndex) {
        return this.rowExpandRatios[rowIndex];
    }

    /**
     * Sets the expand-ratio of given column.
     *
     * <p>
     * The expand ratio defines how excess space is distributed among columns.
     * Excess space means space that is left over from components that are not
     * sized relatively. By default, the excess space is distributed evenly.
     * </p>
     *
     * <p>
     * Note, that the width of this GridLayout needs to be defined (fixed or
     * relative, as opposed to undefined height) for this method to have any
     * effect.
     * <p>
     * Note that checking for relative width for the child components is done on
     * the server so you cannot set a child component to have undefined width on
     * the server and set it to <code>100%</code> in CSS. You must set it to
     * <code>100%</code> on the server.
     *
     * @param colIndex The column index, starting from 0 for the leftmost column.
     * @param ratio    the expand-ratio
     * @see #setWidth(float, Unit)
     */
    public void setColumnExpandRatio(int colIndex, float ratio) {
        this.colExpandRatios[colIndex] = ratio;
        // update client property
        JsonArray colExpandRatios = Json.createArray();
        for (int i = 0; i < this.colExpandRatios.length; i++) {
            colExpandRatios.set(i, this.colExpandRatios[i]);
        }
        this.getElement().setPropertyJson("colExpandRatios", colExpandRatios);
        this.getElement().executeJs("setTimeout(() => this.$connector.update());");
        markAsDirty();
    }

    /**
     * Returns the expand-ratio of given column.
     *
     * @param colIndex The column index, starting from 0 for the leftmost row.
     * @return the expand ratio, 0.0f by default
     * @see #setColumnExpandRatio(int, float)
     */
    public float getColumnExpandRatio(int colIndex) {
        return this.colExpandRatios[colIndex];
    }

    /**
     * Inserts an empty row at the specified position in the grid.
     *
     * @param index Index of the row before which the new row will be inserted.
     *              The topmost row has index 0.
     */
    public void insertRow(int index) {
        if (index > rows) {
            throw new IllegalArgumentException("Cannot insert row at " + index
                    + " in a gridlayout with height: " + rows);
        }
        for (var entry : componentCellMap.entrySet()) {
            Cell existingArea = entry.getValue();
            // areas below the new row must be moved down or stretched
            if (existingArea.y2 >= index) {
                existingArea.y2++;
                if (existingArea.y1 >= index) {
                    existingArea.y1++;
                }
            }
        }
        // if cursor is below the new row, move it down one row
        if (this.cursorY >= index) {
            cursorY++;
        }
        this.setRows(rows + 1);
        markAsDirty();
    }

    /**
     * Sets the number of columns in the grid. The column count can not be
     * reduced if there are any areas that would be outside of the shrunk grid.
     *
     * @param columns the new number of columns in the grid.
     */
    public void setColumns(int columns) {
        // illegal argument
        if (columns < 1) {
            throw new IllegalArgumentException("The number of columns in the grid must be at least 1");
        }
        // no change
        if (this.columns == columns) {
            return;
        }

        if (columns < this.columns) {
            // check that no component is left outside when removing columns
            for (Cell area : componentCellMap.values()) {
                if (area.x2 >= columns) {
                    throw new OutOfBoundsException(area);
                }
            }
        }
        this.columns = columns;
        this.getElement().setProperty("columns", columns);

        updateCellsArray();

        this.colExpandRatios = Arrays.copyOf(this.colExpandRatios, columns);
        JsonArray jsonColExpandRatios = Json.createArray();
        for (int i = 0; i < columns; i++) {
            jsonColExpandRatios.set(i, this.colExpandRatios[i]);
        }
        this.getElement().setPropertyJson("colExpandRatios", jsonColExpandRatios);
    }

    /**
     * Get the number of columns in the grid.
     *
     * @return the number of columns in the grid.
     */
    public int getColumns() {
        return this.columns;
    }

    /**
     * Sets the number of rows in the grid. The number of rows can not be
     * reduced if there are any areas that would be outside of the shrunk grid.
     *
     * @param rows the new number of rows in the grid.
     */
    public void setRows(int rows) {
        // illegal argument
        if (rows < 1) {
            throw new IllegalArgumentException("The number of rows in the grid must be at least 1");
        }
        // no change
        if (this.rows == rows) {
            return;
        }

        if (rows < this.rows) {
            // when removing rows, check no component is out of bounds
            for (Cell existingArea : componentCellMap.values()) {
                if (existingArea.y2 >= rows) {
                    throw new OutOfBoundsException(existingArea);
                }
            }
        }
        this.rows = rows;
        this.getElement().setProperty("rows", rows);

        updateCellsArray();

        this.rowExpandRatios = Arrays.copyOf(rowExpandRatios, rows);
        JsonArray jsonRowExpandRatios = Json.createArray();
        for (int i = 0; i < rows; i++) {
            jsonRowExpandRatios.set(i, this.rowExpandRatios[i]);
        }
        this.getElement().setPropertyJson("rowExpandRatios", jsonRowExpandRatios);
    }

    /**
     * Get the number of rows in the grid.
     *
     * @return the number of rows in the grid.
     */
    public int getRows() {
        return this.rows;
    }

    /**
     * Gets the current x-position (column) of the cursor.
     *
     * <p>
     * The cursor position points the position for the next component that is
     * added without specifying its coordinates (grid cell). When the cursor
     * position is occupied, the next component will be added to first free
     * position after the cursor.
     * </p>
     *
     * @return the grid column the cursor is on, starting from 0.
     */
    public int getCursorX() {
        return cursorX;
    }

    /**
     * Sets the current cursor x-position. This is usually handled automatically
     * by GridLayout.
     *
     * @param cursorX current cursor x-position.
     * @throws IndexOutOfBoundsException if the cursor is set out of the grid.
     */
    public void setCursorX(int cursorX) throws IndexOutOfBoundsException {
        if (cursorX < 0 || cursorX >= getColumns()) {
            throw new IndexOutOfBoundsException("Cursor's X coordinate cannot be set at :" + cursorX + " on a grid with " + getColumns() + " columns.");
        }
        this.cursorX = cursorX;
    }

    /**
     * Gets the current y-position (row) of the cursor.
     *
     * <p>
     * The cursor position points the position for the next component that is
     * added without specifying its coordinates (grid cell). When the cursor
     * position is occupied, the next component will be added to the first free
     * position after the cursor.
     * </p>
     *
     * @return the grid row the Cursor is on.
     */
    public int getCursorY() {
        return cursorY;
    }

    /**
     * Sets the current y-coordinate (row) of the cursor. This is usually
     * handled automatically by GridLayout.
     *
     * @param cursorY the row number, starting from 0 for the topmost row.
     * @throws IndexOutOfBoundsException if the cursor is set out of the grid.
     */
    public void setCursorY(int cursorY) throws IndexOutOfBoundsException {
        if (cursorY < 0 || cursorY >= getRows()) {
            throw new IndexOutOfBoundsException("Cursor's Y coordinate cannot be set at :" + cursorY + " on a grid with " + getRows() + " rows.");
        }
        this.cursorY = cursorY;
    }

    /**
     * Inserts an empty row at the end of the grid.
     */
    public void appendRow() {
        this.insertRow(cursorY + 1);
    }

    /**
     * Removes a row and all the components in the row.
     *
     * <p>
     * Components which span over several rows are removed if the selected row
     * is on the first row of such a component.
     * </p>
     *
     * <p>
     * If the last row is removed then all remaining components will be removed
     * and the grid will be reduced to one row. The cursor will be moved to the
     * upper left cell of the grid.
     * </p>
     *
     * @param row Index of the row to remove. The topmost row has index 0.
     */
    public void removeRow(int row) {
        if (row < 0 || row >= getRows()) {
            throw new IllegalArgumentException("Cannot delete row " + row + " from a grid-layout with height " + getRows());
        }

        // remove all components in row
        for (int col = 0; col < getColumns(); col++) {
            removeComponent(col, row);
        }

        for (Cell cell : componentCellMap.values()) {
            if (cell.y2 >= row) {
                // shrink cells in the selected row
                cell.y2--;

                // if cell is below the removed row, move it up one row
                if (cell.y1 > row) {
                    cell.y1--;
                }
            }
        }
        /*
        Removing the last remaining row means that the dimensions of the grid
        layout will be truncated to 1 empty row and the cursor is moved to the
        first cell.
         */
        if (rows == 1) {
            cursorX = 0;
            cursorY = 0;
        } else {
            // remove the last row
            setRows(rows - 1);
        }

        markAsDirty();
    }

    /**
     * Moves the cursor forward by one. If the cursor goes out of the right grid
     * border, it is moved to the first column of the next row. This action will
     * not move the cursor out of the end of the grid.
     *
     * @see #newLine()
     */
    public void space() {
        cursorX++;
        if (cursorX >= columns) {
            cursorX = 0;
            cursorY++;
        }
        // put cursor back to the end of the grid
        if (cursorY >= rows) {
            cursorX = columns - 1;
            cursorY = rows - 1;
        }
    }

    /**
     * Forces the next component to be added at the beginning of the next line.
     *
     * <p>
     * Sets the cursor column to 0 and increments the cursor row by one.
     * </p>
     *
     * <p>
     * By calling this function you can ensure that no more components are added
     * right of the previous component.
     * </p>
     *
     * @see #space()
     */
    public void newLine() {
        if (cursorY < rows - 1) {
            cursorX = 0;
            cursorY++;
        }
    }

    @Override
    public void setSpacing(boolean spacing) {
        this.spacing = spacing;
        if (spacing) {
            this.getElement().setAttribute("spacing", true);
        } else {
            this.getElement().removeAttribute("spacing");
        }
        this.getElement().executeJs("setTimeout(() => this.$connector.update());");
    }

    @Override
    public boolean isSpacing() {
        return this.spacing;
    }

    @Override
    public Stream<Component> getChildren() {
        return components.stream();
    }

    @Override
    public void replaceComponent(Component oldComponent, Component newComponent) {
        Cell oldCell = componentCellMap.get(oldComponent);
        Cell newCell = componentCellMap.get(newComponent);

        if (oldCell == null) {
            // old component is not in the grid
            addComponent(newComponent);
        } else if (newCell == null) {
            // old component is in the grid, but new one is not
            int index = this.components.indexOf(oldComponent);
            this.components.add(index, newComponent);
            this.components.remove(oldComponent);
            Cell cell = this.componentCellMap.get(oldComponent);
            cell.setContent(newComponent);
            this.componentCellMap.remove(oldComponent);
            this.componentCellMap.put(newComponent, cell);
        } else {
            // both components are in the grid, so just swap their places
            Alignment oldAlignment = oldCell.getAlignment();
            oldCell.setContent(newComponent);
            newCell.setContent(oldComponent);

            oldCell.setAlignment(newCell.getAlignment());
            newCell.setAlignment(oldAlignment);
            componentCellMap.replace(newComponent, oldCell);
            componentCellMap.replace(oldComponent, newCell);
        }
        this.getElement().executeJs("setTimeout(() => this.$connector.update());");
    }

    @Override
    public int getComponentCount() {
        return this.components.size();
    }

    @Override
    public Iterator<Component> iterator() {
        return Collections.unmodifiableCollection(this.components).iterator();
    }

    private void updateCellsArray() {
        if (this.cells == null) {
            this.cells = new Cell[columns][rows];
        } else if (cells.length != columns || cells[0].length != rows) {
            Cell[][] newCells = new Cell[columns][rows];
            for (int i = 0; i < cells.length; i++) {
                for (int j = 0; j < cells[i].length; j++) {
                    if (i < columns && j < rows) {
                        newCells[i][j] = cells[i][j];
                    }
                }
            }
            cells = newCells;
        }
    }

    /**
     * Tests if the given area overlaps with any of the items already on the
     * grid.
     *
     * @param area the Area to be checked for overlapping.
     * @throws OverlapsException if <code>area</code> overlaps with any existing area.
     */
    private void checkExistingOverlaps(Cell area) throws OverlapsException {
        for (Cell existingArea : componentCellMap.values()) {
            if (areasOverlap(area, existingArea)) {
                throw new OverlapsException(existingArea);
            }
        }
    }

    private static boolean areasOverlap(Cell a, Cell b) {
        return a.x1 <= b.x2 && a.y1 <= b.y2 && a.x2 >= b.x1 && a.y2 >= b.y1;
    }

    /**
     * Defines a rectangular area of cells in a GridLayout.
     *
     * <p>
     * Also maintains a reference to the component contained in the area.
     * </p>
     *
     * <p>
     * The area is specified by the cell coordinates of its upper left corner
     * (column1,row1) and lower right corner (column2,row2). As otherwise with
     * GridLayout, the column and row coordinates start from zero.
     * </p>
     */
    public class Area implements Serializable {
        private final ChildComponentData childData;
        private final Component component;

        /**
         * <p>
         * Construct a new area on a grid.
         * </p>
         *
         * @param component the component connected to the area.
         * @param column1   The column of the upper left corner cell of the area. The
         *                  leftmost column has index 0.
         * @param row1      The row of the upper left corner cell of the area. The
         *                  topmost row has index 0.
         * @param column2   The column of the lower right corner cell of the area. The
         *                  leftmost column has index 0.
         * @param row2      The row of the lower right corner cell of the area. The
         *                  topmost row has index 0.
         */
        public Area(Component component, int column1, int row1, int column2,
                    int row2) {
            this.component = component;
            childData = new ChildComponentData();
            childData.alignment = getDefaultComponentAlignment().getBitMask();
            childData.column1 = column1;
            childData.row1 = row1;
            childData.column2 = column2;
            childData.row2 = row2;
        }

        public Area(ChildComponentData childData, Component component) {
            this.childData = childData;
            this.component = component;
        }

        /**
         * Tests if this Area overlaps with another Area.
         *
         * @param other the other Area that is to be tested for overlap with this
         *              area
         * @return <code>true</code> if <code>other</code> area overlaps with
         * this on, <code>false</code> if it does not.
         */
        public boolean overlaps(Area other) {
            ChildComponentData a = this.childData;
            ChildComponentData b = other.childData;
            return a.column1 <= b.column2 && a.row1 <= b.row2
                    && a.column2 >= b.column1 && a.row2 >= b.row1;
        }

        /**
         * Gets the component connected to the area.
         *
         * @return the Component.
         */
        public Component getComponent() {
            return component;
        }

        /**
         * Gets the column of the top-left corner cell.
         *
         * @return the column of the top-left corner cell.
         */
        public int getColumn1() {
            return childData.column1;
        }

        /**
         * Gets the column of the bottom-right corner cell.
         *
         * @return the column of the bottom-right corner cell.
         */
        public int getColumn2() {
            return childData.column2;
        }

        /**
         * Gets the row of the top-left corner cell.
         *
         * @return the row of the top-left corner cell.
         */
        public int getRow1() {
            return childData.row1;
        }

        /**
         * Gets the row of the bottom-right corner cell.
         *
         * @return the row of the bottom-right corner cell.
         */
        public int getRow2() {
            return childData.row2;
        }

        @Override
        public String toString() {
            return String.format("Area{%s,%s - %s,%s}", getColumn1(), getRow1(),
                    getColumn2(), getRow2());
        }
    }

    /**
     * Gridlayout does not support laying components on top of each other. An
     * <code>OverlapsException</code> is thrown when a component already exists (even partly) at the
     * same space on a grid with the new component.
     */
    public static class OverlapsException extends java.lang.RuntimeException {

        private final Cell existingArea;

        /**
         * Constructs an <code>OverlapsException</code>.
         *
         * @param existingArea existing area
         */
        private OverlapsException(Cell existingArea) {
            this.existingArea = existingArea;
        }

        @Override
        public String getMessage() {
            StringBuilder sb = new StringBuilder();
            Component component = existingArea.getContent();
            sb.append(component);
            sb.append("( type = ");
            sb.append(component.getClass().getName());
            sb.append(")");
            sb.append(" is already added to ");
            sb.append(existingArea.y1);
            sb.append(",");
            sb.append(existingArea.x1);
            sb.append(",");
            sb.append(existingArea.y2);
            sb.append(",");
            sb.append(existingArea.x2);
            sb.append("(row1, column1, row2, column2).");

            return sb.toString();
        }

        /**
         * Gets the area .
         *
         * @return the existing area.
         */
        public Cell getArea() {
            return existingArea;
        }
    }

    /**
     * An <code>Exception</code> object which is thrown when an area exceeds the bounds of the grid.
     */
    public static class OutOfBoundsException extends java.lang.RuntimeException {

        private final Cell areaOutOfBounds;

        /**
         * Constructs an <code>OoutOfBoundsException</code> with the specified detail message.
         *
         * @param areaOutOfBounds area out of bounds
         */
        private OutOfBoundsException(Cell areaOutOfBounds) {
            this.areaOutOfBounds = areaOutOfBounds;
        }

        /**
         * Gets the area that is out of bounds.
         *
         * @return the area out of Bound.
         */
        public Cell getArea() {
            return areaOutOfBounds;
        }
    }

    @Tag(Tag.DIV)
    class Cell extends AbstractSingleComponentContainer {
        int x1, y1, x2, y2;
        Alignment alignment;

        Cell(int x1, int y1, Component child) {
            this.x1 = this.x2 = x1;
            this.y1 = this.y2 = y1;
            updateCoordinates(x1, y1, x2, y2);
            this.setPrimaryStyleName(PRIMARY_STYLE_NAME + "-slot");
            this.setContent(child);
        }

        Cell(Component child, int x1, int y1, int x2, int y2) {
            this.x1 = x1;
            this.y1 = y1;
            this.x2 = x2;
            this.y2 = y2;
            updateCoordinates(x1, y1, x2, y2);
            this.setPrimaryStyleName(PRIMARY_STYLE_NAME + "-slot");
            this.setContent(child);
        }

        private void updateCoordinates(int x1, int y1, int x2, int y2) {
            this.getElement().setProperty("x1", x1);
            this.getElement().setProperty("y1", y1);
            this.getElement().setProperty("x2", x2);
            this.getElement().setProperty("y2", y2);
        }

        void setAlignment(Alignment alignment) {
            this.alignment = alignment;
            // if the alignment is something other than topLeft, then we need to
            // align the component inside this cell
            if (alignment != null && (!alignment.isLeft() || !alignment.isTop())) {
                this.getElement().setProperty("horizontalAlignment", alignment.getHorizontalAlignment());
                this.getElement().setProperty("verticalAlignment", alignment.getVerticalAlignment());
                getContent().getElement().getStyle().set("position", "absolute");
            } else {
                this.getElement().removeProperty("horizontalAlignment");
                this.getElement().removeProperty("verticalAlignment");
                getContent().getElement().getStyle().set("position", "relative");
            }
        }

        Alignment getAlignment() {
            return this.alignment;
        }
    }

}
