/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import com.vaadin.flow.component.ComponentEventListener;

/**
 * Legacy version of Listener interface that resembles Vaadin 7/8's
 * <code>Component.Listener</code> API as closely as possible in order to
 * facilitate migration to newer versions of Vaadin.
 *
 * Listener interface for receiving {@link com.vaadin.classic.v8.ui.Event}s.
 *
 * <p>
 * Listener interfaces are the basis of all user interaction handling in Vaadin.
 * You have or create a listener object that receives the events. All event
 * types have their corresponding listener types; they are not, however,
 * required to inherit the {@code Listener} interface, and they rarely do so.
 * </p>
 *
 * <p>
 * This generic listener interface is useful typically when you wish to handle
 * events from different component types in a single listener method
 * ({@code componentEvent()}. If you handle component events in an anonymous
 * listener class, you normally use the component specific listener class.
 * </p>
 *
 * <pre>
 * class Listening extends CustomComponent implements Listener {
 *     Button ok; // Stored for determining the source of an event
 *
 *     Label status; // For displaying info about the event
 *
 *     public Listening() {
 *         VerticalLayout layout = new VerticalLayout();
 *
 *         // Some miscellaneous component
 *         TextField name = new TextField(&quot;Say it all here&quot;);
 *         name.addListener(this);
 *         layout.addComponent(name);
 *
 *         // Handle button clicks as generic events instead
 *         // of Button.ClickEvent events
 *         ok = new Button(&quot;OK&quot;);
 *         ok.addListener(this);
 *         layout.addComponent(ok);
 *
 *         // For displaying information about an event
 *         status = new Label(&quot;&quot;);
 *         layout.addComponent(status);
 *
 *         setCompositionRoot(layout);
 *     }
 *
 *     public void componentEvent(Event event) {
 *         // Act according to the source of the event
 *         if (event.getSource() == ok
 *                 &amp;&amp; event.getClass() == Button.ClickEvent.class)
 *             getWindow().showNotification(&quot;Click!&quot;);
 *
 *         // Display source component and event class names
 *         status.setValue(
 *                 &quot;Event from &quot; + event.getSource().getClass().getName() + &quot;: &quot;
 *                         + event.getClass().getName());
 *     }
 * }
 *
 * Listening listening = new Listening();
 * layout.addComponent(listening);
 * </pre>
 *
 * @see AbstractComponent#addListener(Listener)
 */
public interface Listener extends ComponentEventListener<Event> {
    /**
     * Notifies the listener of a component event.
     *
     * <p>
     * As the event can typically come from one of many source components, you
     * may need to differentiate between the event source by component
     * reference, class, etc.
     * </p>
     *
     * <pre>
     * public void componentEvent(Event event) {
     *     // Act according to the source of the event
     *     if (event.getSource() == ok
     *             &amp;&amp; event.getClass() == Button.ClickEvent.class)
     *         getWindow().showNotification(&quot;Click!&quot;);
     *
     *     // Display source component and event class names
     *     status.setValue(&quot;Event from &quot; + event.getSource().getClass().getName()
     *             + &quot;: &quot; + event.getClass().getName());
     * }
     * </pre>
     *
     * @param event
     *            the event that has occurred.
     */
    void componentEvent(Event event);

    @Override
    default void onComponentEvent(Event event) {
        componentEvent(event);
    }
}
