/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;

import com.vaadin.classic.v8.event.Action;
import com.vaadin.classic.v8.event.MouseEvents;
import com.vaadin.classic.v8.server.Helpers;
import com.vaadin.classic.v8.server.Resource;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.component.orderedlayout.Scroller;
import com.vaadin.flow.dom.DomEvent;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.dom.ElementFactory;
import com.vaadin.flow.internal.nodefeature.ElementPropertyMap;
import com.vaadin.flow.internal.nodefeature.NodeFeature;
import com.vaadin.flow.shared.Registration;

/**
 * Legacy version of Panel - a simple single component container.
 *
 * @author Vaadin Ltd.
 */
@CssImport("./panel.css")
@Tag(Tag.DIV)
public class Panel extends AbstractSingleComponentContainer
        implements Focusable, Action.Notifier {

    private static final String V_PANEL = "v-panel";
    private static final String V_CONTENT = V_PANEL + "-content";
    private static final String V_NO_CAPTION = V_PANEL + "-nocaption";
    private static final String V_CAPTION = V_PANEL + "-caption";
    private static final String V_SCROLLABLE = "v-scrollable";
    private static final String V_CAPTION_WRAP = V_PANEL + "-captionwrap";
    private static final int SCROLL_EVENT_DEBOUNCE_TIMEOUT = 300;

    private String caption;
    private boolean isCaptionAsHtml;
    private final Element captionElement;
    private final Element captionContent;

    private Icon icon;

    private final Element scrollable;
    private boolean isContextMenuPrevented = false;

    /**
     * Creates a new empty panel.
     */
    public Panel() {
        setPrimaryStyleName(V_PANEL);
        // Caption
        captionContent = ElementFactory.createSpan();

        captionElement = ElementFactory.createDiv();
        captionElement.appendChild(captionContent);

        Element captionWrap = ElementFactory.createDiv();
        captionWrap.appendChild(captionElement);
        captionWrap.getClassList().set(V_CAPTION_WRAP, true);

        // Content
        scrollable = new Element("vaadin-scroller");
        /*
         * Create the component for the scroller, but avoid it being mapped from
         * the element to the component. Doing so, any component set with
         * setContent, will have the Panel as parent instead of the scroller
         */
        ComponentUtil.componentFromElement(scrollable, Scroller.class, false);
        scrollable.addEventListener("scroll", this::scrollEventListener)
                .debounce(SCROLL_EVENT_DEBOUNCE_TIMEOUT)
                .addEventData("element.scrollTop")
                .addEventData("element.scrollLeft");

        // Append caption and scroller to panel
        getElement().appendChild(captionWrap);
        setContentWrapper(scrollable);

        // Apply custom styles to caption and scroller
        writeCustomStylesToInternalElements();

        setWidth(100, Unit.PERCENTAGE);
    }

    /**
     * Creates a new empty panel which contains the given content.
     *
     * @param content
     *            the content for the panel.
     */
    public Panel(Component content) {
        this();
        setContent(content);
    }

    /**
     * Creates a new empty panel with caption.
     *
     * @param caption
     *            the caption used in the panel (HTML).
     */
    public Panel(String caption) {
        this();
        setCaption(caption);
    }

    /**
     * Creates a new empty panel with the given caption and content.
     *
     * @param caption
     *            the caption of the panel (HTML).
     * @param content
     *            the content used in the panel.
     */
    public Panel(String caption, Component content) {
        this();
        setCaption(caption);
        setContent(content);
    }

    @Override
    public void beforeClientResponse(boolean initial) {
        super.beforeClientResponse(initial);

        writeCustomStylesToInternalElements();
    }

    /**
     * Add a click listener to the Panel. The listener is called whenever the
     * user clicks inside the Panel. Also, when the click targets a component
     * inside the Panel, provided the targeted component does not prevent the
     * click event from propagating.
     *
     * @see Registration
     *
     * @param listener
     *            The listener to add, not null
     * @return a registration object for removing the listener
     */
    public Registration addClickListener(MouseEvents.ClickListener listener) {
        if (!isContextMenuPrevented) {
            // To replicate FW behavior; when a click listener is added to the
            // Panel component in FW 7/8 the context menu is prevented, even if
            // the registered listener is removed afterwards.
            getElement().addEventListener("contextmenu", evt -> {
            }).addEventData("event.preventDefault()");

            isContextMenuPrevented = true;
        }

        return ComponentUtil.addListener(this, MouseEvents.ClickEvent.class,
                listener);
    }

    private void writeCustomStylesToInternalElements() {
        String captionBaseClass = caption == null || caption.isEmpty()
                ? V_NO_CAPTION
                : V_CAPTION;

        List<String> captionStylesList = new ArrayList<>();
        captionStylesList.add(captionBaseClass);

        List<String> contentStyleList = new ArrayList<>();
        contentStyleList.add(V_CONTENT);

        getCustomStyles().forEach(style -> {
            captionStylesList.add(captionBaseClass + "-" + style);
            contentStyleList.add(V_CONTENT + "-" + style);
        });
        captionElement.getClassList().clear();
        captionElement.getClassList().addAll(captionStylesList);

        contentStyleList.add(V_SCROLLABLE);
        scrollable.getClassList().clear();
        scrollable.getClassList().addAll(contentStyleList);
    }

    /**
     * Sets the caption of the component.
     *
     * <p>
     * A <i>caption</i> is an explanatory textual label accompanying a user
     * interface component, usually shown above, left of, or inside the
     * component.
     * </p>
     *
     * <pre>
     * Panel panel = new Panel();
     * panel.setCaption(&quot;Panel caption&quot;);
     * </pre>
     *
     * <p>
     * The contents of a caption are automatically quoted, so no raw HTML can be
     * rendered in a caption. The validity of the used character encoding,
     * usually UTF-8, is not checked.
     * </p>
     *
     * <p>
     * The caption is displayed inside the component.
     * </p>
     *
     * @param caption
     *            the new caption for the component. If the caption is
     *            {@code null}, no caption is shown, and it does not normally
     *            take any space
     */
    public void setCaption(String caption) {
        this.caption = caption;
        updateCaptionText();
        markAsDirty();
    }

    /**
     * Gets the caption of the component.
     *
     * <p>
     * See {@link #setCaption(String)} for a detailed description of the
     * caption.
     * </p>
     *
     * @return the caption of the component or {@code null} if the caption is
     *         not set.
     * @see #setCaption(String)
     */
    public String getCaption() {
        return caption;
    }

    /**
     * Sets whether the caption is rendered as HTML.
     * <p>
     * If set to true, the captions are rendered in the browser as HTML and the
     * developer is responsible for ensuring no harmful HTML is used. If set to
     * false, the caption is rendered in the browser as plain text.
     * <p>
     * Note: Unlike the Label component in Vaadin 7/8, the script tag is removed
     * when the caption is rendered as html
     *
     * The default is false, i.e. to render that caption as plain text.
     *
     * @param captionAsHtml
     *            true if the captions are rendered as HTML, false if rendered
     *            as plain text
     */
    public void setCaptionAsHtml(boolean captionAsHtml) {
        isCaptionAsHtml = captionAsHtml;
        updateCaptionText();
    }

    private void updateCaptionText() {
        // FW behavior
        String text = caption != null && caption.isEmpty() ? " " : caption;

        if (isCaptionAsHtml()) {
            // Unlike FW, we're removing script tags
            captionContent.setProperty("innerHTML",
                    Jsoup.clean(text, Safelist.relaxed()));
        } else {
            captionContent.setText(text);
        }
    }

    /**
     * Checks whether captions are rendered as HTML
     * <p>
     * The default is false, i.e. to render that caption as plain text.
     *
     * @return true if the captions are rendered as HTML, false if rendered as
     *         plain text
     */
    public boolean isCaptionAsHtml() {
        return isCaptionAsHtml;
    }

    /**
     * Sets the icon of the component.
     *
     * <p>
     * An icon is an explanatory graphical label accompanying a user interface
     * component, usually shown above, left of, or inside the component. Icon is
     * closely related to caption (see {@link #setCaption(String) setCaption()})
     * and is displayed horizontally before..
     * </p>
     *
     * <pre>
     * // Component with an icon from a custom theme
     * Panel panel = new Panel(&quot;Name&quot;);
     * panel.setIcon(VaadinIcon.COPY.create());
     * layout.addComponent(name);
     * </pre>
     *
     * <p>
     * An icon will be rendered inside an HTML element that has the
     * {@code v-icon} CSS style class. The containing layout may enclose an icon
     * and a caption inside elements related to the caption, such as
     * {@code v-caption} .
     * </p>
     *
     * @param icon
     *            the icon of the component. If null, no icon is shown, and it
     *            does not normally take any space.
     * @see #getIcon()
     * @see #setCaption(String)
     */
    public void setIcon(Icon icon) {
        if (this.icon == null) {
            this.icon = icon;
            this.icon.addClassName("v-icon");
            captionElement.insertChild(0, this.icon.getElement());
        } else {
            this.icon = icon;
        }
    }

    /**
     * <p>
     * Non-op method.
     * </p>
     *
     * <p>
     * In order to get the icon set with {@link #setIcon(Icon)}, use
     * {@link #getIconAsIcon()}
     * </p>
     *
     * @see #getIconAsIcon()
     */
    @Override
    public Resource getIcon() {
        return super.getIcon();
    }

    /**
     * Gets the icon resource of the component.
     *
     * <p>
     * See {@link #setIcon(Icon)} for a detailed description of the icon.
     * </p>
     *
     * @return the icon component or {@code null} if the component has no icon
     * @see #setIcon(Icon)
     */
    public Icon getIconAsIcon() {
        return icon;
    }

    /**
     * Gets scroll left offset.
     *
     * <p>
     * Scrolling offset is the number of pixels this scrollable has been
     * scrolled right.
     * </p>
     *
     * @return Horizontal scrolling position in pixels.
     */
    public int getScrollLeft() {
        return scrollable.getProperty("scrollLeft", 0);
    }

    /**
     * Sets scroll left offset.
     *
     * <p>
     * Scrolling offset is the number of pixels this scrollable has been
     * scrolled right.
     * </p>
     *
     * @param scrollLeft
     *            the xOffset.
     */
    public void setScrollLeft(int scrollLeft) {
        scrollable.setProperty("scrollLeft", scrollLeft);
        /*
         * Flow might get lost if the same value is set twice in a row.
         *
         * For instance: (1) server calls panel.setScrollLeft(0) (2) user
         * scrolls then panel on the client (3) server calls
         * panel.setScrollLeft(0) again
         *
         * This is a workaround for such cases, so that it will force the scroll
         * position to be updated on the client.
         */
        scrollable.executeJs("this.scrollLeft=$0", scrollLeft);
    }

    /**
     * Gets scroll top offset.
     *
     * <p>
     * Scrolling offset is the number of pixels this scrollable has been
     * scrolled down.
     * </p>
     *
     * @return Vertical scrolling position in pixels.
     */
    public int getScrollTop() {
        return scrollable.getProperty("scrollTop", 0);
    }

    /**
     * Sets scroll top offset.
     *
     * <p>
     * Scrolling offset is the number of pixels this scrollable has been
     * scrolled down.
     * </p>
     *
     * <p>
     * The scrolling position is limited by the current height of the content
     * area. If the position is below the height, it is scrolled to the bottom.
     * However, if the same response also adds height to the content area,
     * scrolling to bottom only scrolls to the bottom of the previous content
     * area.
     * </p>
     *
     * @param scrollTop
     *            the yOffset.
     */
    public void setScrollTop(int scrollTop) {
        scrollable.setProperty("scrollTop", scrollTop);
        /*
         * Flow might get lost if the same value is set twice in a row.
         *
         * For instance: (1) server calls panel.setScrollTop(0) (2) user scrolls
         * then panel on the client (3) server calls panel.setScrollTop(0) again
         *
         * This is a workaround for such cases, so that it will force the scroll
         * position to be updated on the client.
         */
        scrollable.executeJs("this.scrollTop=$0", scrollTop);
    }

    private void scrollEventListener(DomEvent e) {
        ElementPropertyMap scrollablePropertyMap = scrollable.getNode()
                .getFeature(ElementPropertyMap.class);
        Double scrollTop = (Double) e.getEventData()
                .getNumber("element.scrollTop");
        Double scrollLeft = (Double) e.getEventData()
                .getNumber("element.scrollLeft");

        scrollablePropertyMap.setProperty("scrollTop", scrollTop);
        scrollablePropertyMap.setProperty("scrollLeft", scrollLeft);

        // Removing the changes for the scroll position (if any)
        // to avoid sending the values back to the client
        Map<NodeFeature, Serializable> changes = scrollable.getNode()
                .getChangeTracker(scrollablePropertyMap, () -> null);

        if (changes != null) {
            changes.remove("scrollTop");
            changes.remove("scrollLeft");
        }
    }

    @Override
    public void focus() {
        scrollable.executeJs("setTimeout(function(){$0.focus()},0)",
                scrollable);
    }

    @Override
    public int getTabIndex() {
        return scrollable.getProperty("tabIndex", -1);
    }

    @Override
    public void setTabIndex(int tabIndex) {
        if (tabIndex == -1 && getTabIndex() == -1) {
            /*
             * The scrollable element has tabindex=-1 by default. However, as it
             * is not explicitly set on the element's attribute, it won't be
             * focusable neither by keyboard (which is expected) or
             * programmatically with the focus method. If the user wants to set
             * tabindex to -1 to make it only focusable by calling the focus()
             * method, calling with setProperty alone API won't have any effect
             * because Flow already has this value set to the element and won't
             * send it to the client. This workaround will make sure to also
             * call the JS expression, but only for this scenario.
             */
            scrollable.executeJs("this.tabIndex=$0", tabIndex);
        }
        scrollable.setProperty("tabIndex", tabIndex);
    }

    @Deprecated
    @Override
    public <T extends Action & Action.Listener> void addAction(T action) {
        Helpers.logUnsupportedApiCall(getClass(), "addAction(T)");
    }

    @Deprecated
    @Override
    public <T extends Action & Action.Listener> void removeAction(T action) {
        Helpers.logUnsupportedApiCall(getClass(), "removeAction(T)");
    }

    @Deprecated
    @Override
    public void addActionHandler(Action.Handler actionHandler) {
        Helpers.logUnsupportedApiCall(getClass(),
                "addActionHandler(Action.Handler)");
    }

    @Deprecated
    @Override
    public void removeActionHandler(Action.Handler actionHandler) {
        Helpers.logUnsupportedApiCall(getClass(),
                "removeActionHandler(Action.Handler)");
    }
}
