/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.event;

import java.io.Serializable;
import java.util.Objects;
import java.util.Optional;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.EventData;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.shared.Registration;
import com.vaadin.classic.v8.ui.Label;

/**
 * Wrapper interface for legacy layout events.
 */
public interface LayoutEvents {

    /**
     * Legacy layout click listener interface.
     */
    @FunctionalInterface
    public interface LayoutClickListener
            extends ComponentEventListener<LayoutClickEvent> {

        @Override
        default void onComponentEvent(LayoutClickEvent layoutClickEvent) {
            layoutClick(layoutClickEvent);
        }

        /**
         * Layout has been clicked.
         *
         * @param event
         *            Component click event.
         */
        public void layoutClick(LayoutClickEvent event);
    }

    /**
     * The interface for adding and removing <code>LayoutClickEvent</code>
     * listeners. By implementing this interface a class explicitly announces
     * that it will generate a <code>LayoutClickEvent</code> when a component
     * inside it is clicked and a <code>LayoutClickListener</code> is
     * registered.
     * <p>
     * Note: The general Java convention is not to explicitly declare that a
     * class generates events, but to directly define the
     * <code>addListener</code> and <code>removeListener</code> methods. That
     * way the caller of these methods has no real way of finding out if the
     * class really will send the events, or if it just defines the methods to
     * be able to implement an interface.
     * </p>
     *
     * @see LayoutClickListener
     * @see LayoutClickEvent
     */
    public interface LayoutClickNotifier extends Serializable {
        /**
         * Add a click listener to the layout. The listener is called whenever
         * the user clicks inside the layout. An event is also triggered when
         * the click targets a component inside a nested layout or Panel,
         * provided the targeted component does not prevent the click event from
         * propagating. A caption is not considered part of a component.
         * <p>
         * The child component that was clicked is included in the
         * {@link LayoutClickEvent}.
         *
         * @param listener
         *            The listener to add
         * @return a registration object for removing the listener
         * @see Registration
         */
        public Registration addLayoutClickListener(
                LayoutClickListener listener);
    }

    /**
     * An event fired when the layout has been clicked. The event contains
     * information about the target layout (component) and the child component
     * that was clicked. If no child component was found it is set to null.
     */
    public static class LayoutClickEvent extends MouseEvents.ClickEvent {

        private final Component clickedComponent;
        private final Component childComponent;

        public LayoutClickEvent(Component source, boolean fromClient,
                @EventData("event.clientX") int clientX,
                @EventData("event.clientY") int clientY,
                @EventData("event.detail") int clickCount,
                @EventData("event.button") int button,
                @EventData("event.ctrlKey") boolean ctrlKey,
                @EventData("event.shiftKey") boolean shiftKey,
                @EventData("event.altKey") boolean altKey,
                @EventData("event.metaKey") boolean metaKey,
                @EventData("event.clientX - element.getBoundingClientRect().x") int relativeX,
                @EventData("event.clientY - element.getBoundingClientRect().y") int relativeY,
                @EventData("event.target") Element eventTarget) {
            super(source, fromClient, clientX, clientY, clickCount, button,
                    ctrlKey, shiftKey, altKey, metaKey, relativeX, relativeY);
            clickedComponent = getComponentForElement(eventTarget);
            childComponent = getSourcesDirectChildWith(clickedComponent)
                    .orElse(null);
        }

        /*
         * Since any element that is created via server side (slot) can be
         * mapped, we need to map it to a component manually.
         */
        private static Component getComponentForElement(
                com.vaadin.flow.dom.Element eventTarget) {
            if (eventTarget == null) {
                return null;
            }
            Optional<Component> clickedComponent = ComponentUtil
                    .findParentComponent(eventTarget);

            // FIXME this should not be necessary ...
            if (clickedComponent.flatMap(Component::getParent)
                    .filter(parent -> parent instanceof Label).isPresent()) {
                clickedComponent = clickedComponent.get().getParent();
            }
            return clickedComponent.orElse(null);
        }

        private Optional<Component> getSourcesDirectChildWith(
                Component component) {
            if (component == source) {
                return Optional.empty();
            }
            Optional<Component> potentialDirectChild = Optional.of(component);
            while (potentialDirectChild.flatMap(Component::getParent)
                    .filter(that -> !Objects.equals(that, source))
                    .isPresent()) {
                potentialDirectChild = potentialDirectChild.get().getParent();
            }
            return potentialDirectChild;
        }

        /**
         * Returns the component that was clicked, which is somewhere inside the
         * parent layout on which the listener was registered.
         * <p>
         * For the direct child component of the layout, see
         * {@link #getChildComponent()}.
         *
         * @return clicked {@link Component}, null if none found
         */
        public Component getClickedComponent() {
            return clickedComponent;
        }

        /**
         * Returns the direct child component of the layout which contains the
         * clicked component.
         * <p>
         * For the clicked component inside that child component of the layout,
         * see {@link #getClickedComponent()}.
         *
         * @return direct child {@link Component} of the layout which contains
         *         the clicked Component, null if none found
         */
        public Component getChildComponent() {
            return childComponent;
        }
    }
}
