/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.server;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEvent;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.shared.Registration;
import com.vaadin.flow.component.ComponentUtil;

/**
 * Legacy version of ClientConnector that resembles Vaadin 7/8's ClientConnector
 * API as closely as possible in order to facilitate migration to newer versions
 * of Vaadin.
 * <p>
 * Interface implemented by all legacy connectors that are capable of
 * communicating with the client side.
 *
 * @author Vaadin Ltd
 */
public interface ClientConnector {

    /**
     * Called before the response is written to client side. Gives the legacy
     * connector an opportunity to set computed/dynamic state values.
     * <p>
     * This is implemented for the legacy components to keep consistent behavior
     * and e.g. legaze size calculations working.
     * <p>
     * <em>NOTE:</em> the parameter {@code initial} is {@code false} when the
     * component is attached again to a new {@link com.vaadin.flow.component.UI}
     * when {@link com.vaadin.flow.router.PreserveOnRefresh} is used.
     *
     * @param initial
     *            <code>true</code> if the client-side connector will be created
     *            and initialized after this method has been invoked.
     *            <code>false</code> if there is already an initialized
     *            client-side connector.
     */
    void beforeClientResponse(boolean initial);

    /**
     * Marks that this classic component's state might have changed. When the
     * framework is about to send new data to the client-side, it will run
     * {@link #beforeClientResponse(boolean)} for all legacy components that are
     * marked as dirty.
     */
    void markAsDirty();

    /**
     * Causes this classic component and all child components below it to be
     * marked as dirty.
     * <p>
     * This should only be used in special cases, e.g when the state of a
     * descendant depends on the state of an ancestor.
     * <p>
     * This will go through the whole component subtree, including any
     * non-legacy components.
     *
     * @see #markAsDirty()
     */
    void markAsDirtyRecursive();

    /**
     * Add a listener for legacy connector attach events. This could be migrated
     * to {@link Component#addAttachListener(ComponentEventListener)}.
     *
     * @param listener
     *            the listener to add
     * @return Registration for unregistering the listener
     */
    Registration addLegacyAttachListener(AttachListener listener);

    /**
     * Add a listener for connector detach events. This could be migrated to
     * {@link Component#addDetachListener(ComponentEventListener)}
     *
     * @param listener
     *            the listener to add
     * @return Registration for unregistering the listener
     */
    Registration addLegacyDetachListener(DetachListener listener);

    /**
     * Notifies the connector that it is connected to a VaadinSession (and
     * therefore also to a UI).
     * <p>
     * The caller of this method is
     * {@link ComponentUtil#onComponentAttach(Component, boolean)}} if the
     * parent is itself already attached to the session. If not, it's called
     * when the parent is attached to the session. This method is always called
     * before the connector's data is sent to the client-side for the first
     * time.
     * </p>
     *
     * <p>
     * The attachment logic is implemented in {@link AbstractClientConnector}.
     * </p>
     */
    void attach();

    /**
     * Notifies the connector that it is detached from its VaadinSession.
     *
     * </p>
     */
    void detach();

    /**
     * Checks if the connector is attached to a VaadinSession.
     *
     * @return true if the connector is attached to a session, false otherwise
     */
    boolean isAttached();

    /**
     * Checks if the communicator is enabled. An enabled communicator is allowed
     * to receive messages from its counter-part.
     *
     * @return true if the connector can receive messages, false otherwise
     */
    boolean isConnectorEnabled();

    /**
     * Interface for listening {@link AttachEvent connector attach events}. This
     * can be replaced with
     * {@link ComponentEventListener<com.vaadin.flow.component.AttachEvent>}.
     *
     */
    @FunctionalInterface
    interface AttachListener
            extends ComponentEventListener<ClientConnector.AttachEvent> {

        @Override
        default void onComponentEvent(ClientConnector.AttachEvent event) {
            this.attach(event);
        }

        /**
         * Called when a AttachListener is notified of a AttachEvent.
         *
         * @param event
         *            The attach event that was fired.
         */
        void attach(ClientConnector.AttachEvent event);
    }

    /**
     * Interface for listening {@link DetachEvent connector attach events}. This
     * can be replaced with
     * {@link ComponentEventListener<com.vaadin.flow.component.DetachEvent>}
     *
     */
    @FunctionalInterface
    interface DetachListener
            extends ComponentEventListener<ClientConnector.DetachEvent> {

        @Override
        default void onComponentEvent(DetachEvent event) {
            this.detach(event);
        }

        /**
         * Called when a DetachListener is notified of a DetachEvent.
         *
         * @param event
         *            The detach event that was fired.
         */
        void detach(DetachEvent event);
    }

    /**
     * Event fired before a connector is attached from the application.
     */
    class AttachEvent extends ComponentEvent<Component> {

        /**
         * Creates a new event using the given source and indicator whether the
         * event originated from the client side or the server side.
         *
         * @param source
         *            the source component
         */
        public AttachEvent(Component source) {
            super(source, false);
        }
    }

    /**
     * Event fired before a connector is detached from the application.
     */
    class DetachEvent extends ComponentEvent<Component> {

        /**
         * Creates a new event using the given source and indicator whether the
         * event originated from the client side or the server side.
         *
         * @param source
         *            the source component
         */
        public DetachEvent(Component source) {
            super(source, false);
        }
    }
}
