/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.server;

import java.io.Serializable;
import java.util.Iterator;

import com.vaadin.classic.v8.ui.GridLayout;
import com.vaadin.classic.v8.ui.GridLayout.Area;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.dom.ElementConstants;
import com.vaadin.classic.v8.ui.AbstractComponent;
import com.vaadin.classic.v8.ui.AbstractOrderedLayout;
import com.vaadin.classic.v8.ui.VerticalLayout;

public class ComponentSizeValidator implements Serializable {

    private static boolean hasNonRelativeHeightComponent(
            AbstractOrderedLayout ol) {
        Iterator<Component> it = ol.getChildren().iterator();
        while (it.hasNext()) {
            if (!hasRelativeHeight(it.next())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Internal framework sizing magic method that has never had javadocs. The
     * purpose is to block users from setting relative (%) size child components
     * when inside certain legacy components.
     *
     * @param component
     *            the flow or classic component
     * @return whether the component can define its height or not
     */
    public static boolean parentCanDefineHeight(Component component) {
        if (!component.getParent().isPresent()) {
            // this should never occur when called via AbstractComponent,
            // but cannot be sure about external usage
            return true;
        }
        final Component parent = component.getParent().get();
        if (isUndefinedHeight(parent)) {
            // Undefined height parent
            if (parent instanceof AbstractOrderedLayout) {
                if (parent instanceof VerticalLayout) {
                    return false;
                }
                return hasNonRelativeHeightComponent(
                        (AbstractOrderedLayout) parent);

            } else if (parent instanceof GridLayout) {
                GridLayout gl = (GridLayout) parent;
                Area componentArea = gl.getComponentArea(component);
                for (int row = componentArea.getRow1(); row <= componentArea
                        .getRow2(); row++) {
                    for (int column = 0; column < gl.getColumns(); column++) {
                        Component c = gl.getComponent(column, row);
                        if (c != null) {
                            if (!hasRelativeHeight(c)) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            }
            /*
             * else if (isForm(parent)) {
             */
            /*
             * If some other part of the form is not relative it determines the
             * component width
             */
            /*
             *
             * return formHasNonRelativeWidthComponent(parent); }
             */
            return true; // next part not applicable, yet...
            /*
             * if (parent instanceof Panel || parent instanceof
             * AbstractSplitPanel || parent instanceof TabSheet || parent
             * instanceof CustomComponent) { // height undefined, we know how
             * how component works and no // exceptions return false; } else {
             * // We cannot generally know if undefined component can serve //
             * space for children (like CustomLayout or component built by //
             * third party) so we assume they can return true; }
             */

        } else if (hasRelativeHeight(parent)) {
            // Relative height
            if (parent.getParent().isPresent()) {
                return parentCanDefineHeight(parent);
            }
            return true;
        } else {
            // Absolute height
            return true;
        }
    }

    private static boolean hasRelativeHeight(Component component) {
        if (component instanceof AbstractComponent) {
            return ((AbstractComponent) component).getHeight() > 0
                    && ((AbstractComponent) component)
                            .getHeightUnits() == Unit.PERCENTAGE;
        }
        try {
            final SizeWithUnit sizeWithUnit = SizeWithUnit
                    .parseStringSize(component.getElement().getStyle()
                            .get(ElementConstants.STYLE_HEIGHT));
            return sizeWithUnit != null
                    && sizeWithUnit.getUnit() == Unit.PERCENTAGE
                    && sizeWithUnit.getSize() > 0; // everything else is not
                                                   // relative
        } catch (IllegalArgumentException iae) {
            // legacy code doesn't parse all size units - anyway not relative,
            // though maybe vh & vh should be treated as relative in theory?
            return false;
        }
    }

    private static boolean hasNonRelativeWidthComponent(
            AbstractOrderedLayout ol) {
        Iterator<Component> it = ol.iterator();
        while (it.hasNext()) {
            if (!hasRelativeWidth(it.next())) {
                return true;
            }
        }
        return false;
    }

    private static boolean hasRelativeWidth(Component component) {
        if (component instanceof AbstractComponent) {
            return ((AbstractComponent) component).getWidth() > 0
                    && ((AbstractComponent) component)
                            .getWidthUnits() == Unit.PERCENTAGE;
        }
        try {
            final SizeWithUnit sizeWithUnit = SizeWithUnit
                    .parseStringSize(component.getElement().getStyle()
                            .get(ElementConstants.STYLE_WIDTH));
            return sizeWithUnit != null
                    && sizeWithUnit.getUnit() == Unit.PERCENTAGE
                    && sizeWithUnit.getSize() > 0; // everything else is defined
        } catch (IllegalArgumentException iae) {
            // legacy code doesn't parse all size types, but it is not relative,
            // though maybe vh & vh should be treated as relative in theory?
            return false;
        }
    }

    /**
     * Internal framework sizing magic method that has never had javadocs. The
     * purpose is to block users from setting relative (%) size child components
     * when inside certain classic components.
     *
     * @param component
     *            the flow or classic component
     * @return whether the component can define its width or not
     */
    public static boolean parentCanDefineWidth(Component component) {
        if (!component.getParent().isPresent()) {
            // this should never occur when called via AbstractComponent,
            // but cannot be sure about external usage
            return true;
        }
        final Component parent = component.getParent().get();
        /*
         * if (parent instanceof Window) { // Sub window with undefined size has
         * a min-width return true; }
         */

        if (isUndefinedWidth(parent)) {
            // Undefined width

            if (parent instanceof AbstractOrderedLayout) {
                AbstractOrderedLayout abstractOrderedLayout = (AbstractOrderedLayout) parent;

                // VerticalLayout and a child defines height
                return abstractOrderedLayout instanceof VerticalLayout
                        && hasNonRelativeWidthComponent(abstractOrderedLayout);
            } else if (parent instanceof GridLayout) {
                GridLayout gl = (GridLayout) parent;
                Area componentArea = gl.getComponentArea(component);
                for (int col = componentArea.getColumn1(); col <= componentArea
                        .getColumn2(); col++) {
                    for (int row = 0; row < gl.getRows(); row++) {
                        Component c = gl.getComponent(col, row);
                        if (c != null) {
                            if (!hasRelativeWidth(c)) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            }
            /*
             * else if (parent instanceof AbstractSplitPanel || parent
             * instanceof TabSheet || parent instanceof CustomComponent) {
             * return false; } else if (parent instanceof Window) { // Sub
             * window can define width based on caption return
             * parent.getCaption() != null && !parent.getCaption().isEmpty(); }
             */
            return true; // next part not applicable, yet ...
            // return !(parent instanceof Panel);
        } else if (hasRelativeWidth(parent)) {
            // Relative width

            if (!parent.getParent().isPresent()) {
                return true;
            }

            return parentCanDefineWidth(parent);
        } else {
            return true;
        }

    }

    private static boolean isUndefinedHeight(Component component) {
        if (component instanceof AbstractComponent) {
            return ((AbstractComponent) component).getHeight() < 0;
        }
        try {
            final SizeWithUnit sizeWithUnit = SizeWithUnit
                    .parseStringSize(component.getElement().getStyle()
                            .get(ElementConstants.STYLE_HEIGHT));
            return sizeWithUnit == null; // everything else is defined
        } catch (IllegalArgumentException iae) {
            // legacy code doesn't parse all size types, but it is not undefined
            return false;
        }
    }

    private static boolean isUndefinedWidth(Component component) {
        if (component instanceof AbstractComponent) {
            return ((AbstractComponent) component).getWidth() < 0;
        }
        try {
            final SizeWithUnit sizeWithUnit = SizeWithUnit
                    .parseStringSize(component.getElement().getStyle()
                            .get(ElementConstants.STYLE_WIDTH));
            return sizeWithUnit == null; // everything else is defined
        } catch (IllegalArgumentException iae) {
            // legacy code doesn't parse all size types, but it is not undefined
            return false;
        }
    }

    private ComponentSizeValidator() {
        // utility class
    }

}
