/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.shared;

import java.io.Serializable;

import com.vaadin.classic.v8.server.Helpers;

/**
 * Legacy helper class to store and transfer mouse event details.
 *
 * @author Vaadin Ltd
 */
public class MouseEventDetails implements Serializable {
    /**
     * Constants for mouse buttons.
     */
    public enum MouseButton {
        LEFT("left"), RIGHT("right"), MIDDLE("middle");

        private String name;

        private MouseButton(String name) {
            this.name = name;
        }

        /**
         * Returns a human readable text representing the button.
         *
         * @return
         */
        public String getName() {
            return name;
        }

    }

    private static final char DELIM = ',';
    // From com.google.gwt.user.client.Event
    private static final int ONDBLCLICK = 0x00002;

    private MouseButton button;
    // the following defaults are different from FW but should make no
    // difference
    private int clientX = 0;
    private int clientY = 0;
    private boolean altKey = false;
    private boolean ctrlKey = false;
    private boolean metaKey = false;
    private boolean shiftKey = false;
    private int type;
    private int relativeX = -1;
    private int relativeY = -1;

    public MouseEventDetails() {
    }

    public void setButton(MouseButton button) {
        this.button = button;
    }

    public void setClientX(int clientX) {
        this.clientX = clientX;
    }

    public void setClientY(int clientY) {
        this.clientY = clientY;
    }

    public void setAltKey(boolean altKey) {
        this.altKey = altKey;
    }

    public void setCtrlKey(boolean ctrlKey) {
        this.ctrlKey = ctrlKey;
    }

    public void setMetaKey(boolean metaKey) {
        this.metaKey = metaKey;
    }

    public void setShiftKey(boolean shiftKey) {
        this.shiftKey = shiftKey;
    }

    public void setType(int type) {
        this.type = type;
    }

    public void setRelativeX(int relativeX) {
        this.relativeX = relativeX;
    }

    public void setRelativeY(int relativeY) {
        this.relativeY = relativeY;
    }

    public MouseButton getButton() {
        return button;
    }

    public int getClientX() {
        return clientX;
    }

    public int getClientY() {
        return clientY;
    }

    public String getButtonName() {
        return button == null ? "" : button.getName();
    }

    public int getType() {
        return type;
    }

    public int getRelativeX() {
        return relativeX;
    }

    public int getRelativeY() {
        return relativeY;
    }

    public boolean isAltKey() {
        return altKey;
    }

    public boolean isCtrlKey() {
        return ctrlKey;
    }

    public boolean isMetaKey() {
        return metaKey;
    }

    public boolean isShiftKey() {
        return shiftKey;
    }

    public boolean isDoubleClick() {
        return type == ONDBLCLICK;
    }

    @Override
    public String toString() {
        return serialize();
    }

    public String serialize() {
        return button.toString() + DELIM + clientX + DELIM + clientY + DELIM
                + altKey + DELIM + ctrlKey + DELIM + metaKey + DELIM + shiftKey
                + DELIM + type + DELIM + relativeX + DELIM + relativeY;
    }

    // this is an old legacy API so keeping it for fun's sake
    public static MouseEventDetails deSerialize(String serializedString) {
        Helpers.logUnsupportedApiCall(MouseEventDetails.class,
                "deSerialize(String)");
        MouseEventDetails instance = new MouseEventDetails();
        String[] fields = serializedString.split(",");
        instance.button = MouseButton.valueOf(fields[0]);
        instance.clientX = Integer.parseInt(fields[1]);
        instance.clientY = Integer.parseInt(fields[2]);
        instance.altKey = Boolean.parseBoolean(fields[3]);
        instance.ctrlKey = Boolean.parseBoolean(fields[4]);
        instance.metaKey = Boolean.parseBoolean(fields[5]);
        instance.shiftKey = Boolean.parseBoolean(fields[6]);
        instance.type = Integer.parseInt(fields[7]);
        instance.relativeX = Integer.parseInt(fields[8]);
        instance.relativeY = Integer.parseInt(fields[9]);
        return instance;
    }

}
