/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.classic.v8.event.LayoutEvents;
import com.vaadin.classic.v8.server.SizeWithUnit;
import com.vaadin.classic.v8.server.Sizeable;
import com.vaadin.classic.v8.shared.ui.AlignmentInfo;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.HasSize;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.dom.ClassList;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.dom.ElementFactory;
import com.vaadin.flow.shared.Registration;

/**
 * Legacy version of AbstractOrderedLayout that resembles Vaadin 7/8's
 * AbstractOrderedLayout API as closely as possible in order to facilitate
 * migration to newer versions of Vaadin.
 * <p>
 * Base class for ordered layouts.
 *
 * @author Vaadin Ltd
 */
public abstract class AbstractOrderedLayout extends AbstractLayout
        implements Layout.AlignmentHandler, Layout.MarginHandler,
        LayoutEvents.LayoutClickNotifier {
    private static final String SLOT_CLASS_NAME = "v-slot";
    final static String ALIGN_CLASS_PREFIX = "v-align-";
    private final boolean isVertical;
    private boolean isSpacing = false;
    private MarginInfo marginInfo;
    private Element expandWrapper = null;

    private boolean movingChildrenInternally; // ignore detach completely
    private boolean removingChildrenInternally; // just remove detach listener

    private Alignment defaultComponentAlignment = Alignment.TOP_LEFT;

    /**
     * Constructs an ordered layout with the indicated orientation.
     *
     * @param isVertical
     *            {@code true} if it should be vertically oriented,
     *            {@code false} for horizontally oriented
     */
    public AbstractOrderedLayout(boolean isVertical) {
        this.isVertical = isVertical;
        addInternalStyles("v-layout",
                isVertical ? "v-vertical" : "v-horizontal");
    }

    /**
     * Constructs an ordered layout with the horizontal orientation.
     */
    public AbstractOrderedLayout() {
        this(false);
    }

    @Override
    public void beforeClientResponse(boolean initial) {
        setMarginToElement(); // needs to be called before, so it adds/removes
        // the internal style
        super.beforeClientResponse(initial);

        cleanupSlotsAndSpacing();
        setSlotClasses();
        setExpandRatiosToSlots();
    }

    @Override
    public void addComponent(Component component) {
        if (component.getParent().isPresent()) {
            AbstractSingleComponentContainer.removeFromParent(component);
        }
        if (component instanceof AbstractComponent) {
            ((AbstractComponent) component).setParent(null);
        }

        Element container = getContainer();

        if (isSpacing && container.getChildCount() > 0) {
            container.appendChild(createSpacing());
        }
        container.appendChild(wrapInSlot(component));

        setComponentAlignment(component, getDefaultComponentAlignment());
        addDetachListener(component);
        // in case the component was already a child
        cleanupSlotsAndSpacing();
        fireComponentAttachEvent(component);
        markAsDirtyRecursive();
    }

    private void addDetachListener(Component component) {
        // one-time detach listener added for clearing the slot always
        AtomicReference<Registration> holder = new AtomicReference<>();
        holder.set(component.getElement().addDetachListener(event -> {
            if (movingChildrenInternally) {
                return;
            }
            if (!removingChildrenInternally) {
                cleanupSlotsAndSpacing(); // done already by remove-methods
                setExpandRatiosToSlots();
            }
            markAsDirtyRecursive();
            holder.get().remove();
        }));
    }

    @Override
    public void add(Component... components) {
        addComponents(components);
    }

    /**
     * Adds the given components to this layout and sets them as expanded. The
     * main layout dimension is set to 100% if it is currently undefined.
     * <p>
     * The components are added in the provided order to the end of this layout.
     * Any components that are already children of this layout will be moved to
     * new positions.
     *
     * @param components
     *            the components to set, not <code>null</code>
     */
    public void addComponentsAndExpand(Component... components) {
        addComponents(components);

        if (isVertical) {
            if (getHeight() < 0) {
                setHeight(100, Unit.PERCENTAGE);
            }
        } else if (getWidth() < 0) {
            setWidth(100, Unit.PERCENTAGE);
        }

        for (Component component : components) {
            if (isVertical) {
                if (component instanceof HasSize) {
                    ((HasSize) component).setHeight(100,
                            com.vaadin.flow.component.Unit.PERCENTAGE);
                }
                if (component instanceof Sizeable) {
                    ((Sizeable) component).setHeight(100, Unit.PERCENTAGE);
                }
            } else {
                if (component instanceof HasSize) {
                    ((HasSize) component).setWidth(100,
                            com.vaadin.flow.component.Unit.PERCENTAGE);
                }
                if (component instanceof Sizeable) {
                    ((Sizeable) component).setWidth(100, Unit.PERCENTAGE);
                }
            }
            setExpandRatio(component, 1);
        }
    }

    private void cleanupSlotsAndSpacing() {
        AtomicInteger index = new AtomicInteger();
        int lastComponentIndex = (int) (getChildren().count() * 2 - 2);

        // need to use a copy due to removing
        getContainer().getChildren().collect(Collectors.toList())
                .forEach(childElement -> {
                    if ((isSpacing
                            && childElement.getClassList().contains("v-spacing")
                            && index.get() % 2 == 1
                            && index.get() < lastComponentIndex)
                            || childElement.getChildCount() > 0) {
                        index.incrementAndGet();
                    } else {
                        childElement.removeFromParent();
                    }
                });
    }

    @Override
    protected void doSetHeight(SizeWithUnit height) {
        super.doSetHeight(height);

        if (!isVertical) {
            return;
        }

        if (getHeight() > 0.0) {
            addExpandWrapper();
        } else {
            removeExpandWrapper();
        }
    }

    @Override
    protected void doSetWidth(SizeWithUnit width) {
        super.doSetWidth(width);

        if (isVertical) {
            return;
        }

        if (getWidth() > 0.0) {
            addExpandWrapper();
        } else {
            removeExpandWrapper();
        }
    }

    /**
     * Enable spacing between child components within this layout.
     *
     * @param spacing
     *            true if spacing should be turned on, false if it should be
     *            turned off
     */
    public void setSpacing(boolean spacing) {
        if (spacing != isSpacing) {
            isSpacing = spacing;

            if (isSpacing) {
                addSpacingElements();
            } else {
                removeSpacingElements();
            }
        }
    }

    /**
     * get the state of spacing in this layout
     *
     * @return true if spacing between child components within this layout is
     *         enabled, false otherwise
     */
    public boolean isSpacing() {
        return isSpacing;
    }

    /**
     * (non-Javadoc)
     *
     * @see Layout.MarginHandler#setMargin(boolean)
     */
    @Override
    public void setMargin(boolean enabled) {
        setMargin(new MarginInfo(enabled));
    }

    /**
     * (non-Javadoc)
     *
     * @see Layout.MarginHandler#setMargin(MarginInfo)
     */
    @Override
    public void setMargin(MarginInfo marginInfo) {
        this.marginInfo = marginInfo;
        markAsDirty();
    }

    /**
     * (non-Javadoc)
     *
     * @see Layout.MarginHandler#getMargin()
     */
    @Override
    public MarginInfo getMargin() {
        if (marginInfo == null) {
            marginInfo = new MarginInfo(false);
        }
        return marginInfo;
    }

    private void setMarginToElement() {
        if (marginInfo == null) {
            return;
        }

        toggleInternalStyle("v-margin-top", marginInfo.hasTop());
        toggleInternalStyle("v-margin-bottom", marginInfo.hasBottom());
        toggleInternalStyle("v-margin-left", marginInfo.hasLeft());
        toggleInternalStyle("v-margin-right", marginInfo.hasRight());
    }

    private void toggleInternalStyle(String style, boolean add) {
        if (add) {
            addInternalStyles(style);
        } else {
            removeInternalStyles(style);
        }
    }

    @Override
    public void remove(Component... components) {
        Objects.requireNonNull(components, "Components should not be null");
        for (Component component : components) {
            removeComponent(component);
        }
    }

    @Override
    public void removeComponent(Component component) {
        // legacy behavior for non-children is NOOP (different from flow)
        if (!isChild(component)) {
            return;
        }
        removingChildrenInternally = true;
        doRemoveComponent(component);
        removingChildrenInternally = false;
        markAsDirtyRecursive();
    }

    private void doRemoveComponent(Component component) {
        final Element element = component.getElement();
        final Element slot = element.getParent();
        final Element container = getContainer();
        final int slotIndex = container.indexOfChild(slot);
        final int spacingIndex = slotIndex < (container.getChildCount() - 1)
                ? slotIndex
                : (slotIndex - 1);
        // cleanup eagerly
        // order matters - detach component from slot first so its parent is
        // null
        element.removeFromParent();
        slot.removeFromParent();
        if (isSpacing && spacingIndex > -1) {
            container.removeChild(spacingIndex);
        }
        fireComponentDetachEvent(component);
    }

    private Element wrapInSlot(Component component) {
        Element slot = ElementFactory.createDiv();
        slot.setAttribute("class", SLOT_CLASS_NAME);
        slot.appendChild(component.getElement());
        return slot;
    }

    /**
     * <p>
     * This method is used to control how excess space in layout is distributed
     * among components. Excess space may exist if layout is sized and contained
     * non relatively sized components don't consume all available space.
     *
     * <p>
     * Example how to distribute 1:3 (33%) for component1 and 2:3 (67%) for
     * component2 :
     *
     * <code>
     * layout.setExpandRatio(component1, 1);<br>
     * layout.setExpandRatio(component2, 2);
     * </code>
     *
     * <p>
     * If no ratios have been set, the excess space is distributed evenly among
     * all components.
     *
     * <p>
     * Note, that width or height (depending on orientation) needs to be defined
     * for this method to have any effect.
     *
     * @param component
     *            the component in this layout which expand ratio is to be set
     * @param ratio
     *            new expand ratio (greater or equal to 0)
     * @throws IllegalArgumentException
     *             if the expand ratio is negative or the component is not a
     *             direct child of the layout
     */
    public void setExpandRatio(Component component, float ratio) {
        if (!isChild(component)) {
            throw new IllegalArgumentException(
                    "The given component is not a child of this layout");
        }
        if (ratio < 0.0f) {
            throw new IllegalArgumentException(
                    "Expand ratio can't be less than 0.0");
        }
        ComponentUtil.setData(component, ExpandRatioKey.class,
                ratio > 0.0f ? new ExpandRatioKey(ratio) : null);
        setExpandRatiosToSlots();
    }

    /**
     * Returns the expand ratio of given component.
     *
     * @param component
     *            which expand ratios is requested
     * @return expand ratio of given component, 0.0f by default.
     */
    public float getExpandRatio(Component component) {
        if (!isChild(component)) {
            throw new IllegalArgumentException(
                    "The given component is not a child of this layout");
        }
        ExpandRatioKey expandRatio = ComponentUtil.getData(component,
                ExpandRatioKey.class);
        return expandRatio != null ? expandRatio.value : 0.0f;
    }

    private boolean isChild(Component component) {
        return component.getParent().isPresent()
                && component.getParent().get() == this;
    }

    private void setExpandRatiosToSlots() {
        boolean hasExpandRatio = getChildren()
                .anyMatch(component -> getExpandRatio(component) > 0.0f);

        getChildren().forEach(component -> {
            Element slot = component.getElement().getParent();
            if (hasExpandRatio) {
                float expandRatio = getExpandRatio(component);
                slot.getStyle().set("flex-grow", String.valueOf(expandRatio));
            } else {
                slot.getStyle().remove("flex-grow");
            }
        });
    }

    private void addExpandWrapper() {
        // TODO: with addition of expand ratio related methods, additional logic
        // might be needed for adding/removing the v-expand wrapper
        // Note: adding the expand wrapper should recover previously set expand
        // ratios.
        // i.e. this component should "remember" ratio of each of the added
        // components
        // https://github.com/vaadin/flow-legacy-components/issues/22
        if (expandWrapper == null) {
            expandWrapper = ElementFactory.createDiv();
            expandWrapper.getClassList().add("v-expand");
            movingChildrenInternally = true;
            expandWrapper.appendChild(
                    getElement().getChildren().toArray(Element[]::new));
            getElement().appendChild(expandWrapper);
            movingChildrenInternally = false;
        }
    }

    private void removeExpandWrapper() {
        // TODO: with addition of expand ratio related methods, additional logic
        // might be needed for adding/removing the v-expand wrapper
        // https://github.com/vaadin/flow-legacy-components/issues/22
        if (expandWrapper != null) {
            movingChildrenInternally = true;
            // due to a flow issue, removal needs to happen first
            expandWrapper.removeFromParent();
            getElement().appendChild(
                    expandWrapper.getChildren().toArray(Element[]::new));
            expandWrapper = null;
            movingChildrenInternally = false;
        }
    }

    private void removeSpacingElements() {
        getContainer().getChildren().collect(Collectors.toList()).stream()
                .filter(child -> child.getClassList().contains("v-spacing"))
                .forEach(Element::removeFromParent);
    }

    private void addSpacingElements() {
        for (int i = 1; i < getContainer().getChildCount();) {
            getContainer().insertChild(i, createSpacing());
            i = i + 2;
        }
    }

    private Element createSpacing() {
        final Element div = ElementFactory.createDiv();
        div.getClassList().add("v-spacing");
        return div;
    }

    private Element getContainer() {
        return expandWrapper == null ? getElement() : expandWrapper;
    }

    @Override
    public Stream<Component> getChildren() {
        Element container = expandWrapper == null ? getElement()
                : expandWrapper;
        return container.getChildren()
                .filter(child -> child.getChildCount() > 0) // spacing
                .map(element -> element.getChild(0)).map(Element::getComponent)
                .map(Optional::get);
    }

    /**
     * Delegates to {@link #addComponent(Component, int)}.
     */
    @Override
    public void addComponentAtIndex(int index, Component component) {
        addComponent(component, index);
    }

    /**
     * Adds a component into indexed position in this container.
     *
     * @param component
     *            the component to be added.
     * @param index
     *            the index of the component position. The components currently
     *            in and after the position are shifted forwards.
     */
    public void addComponent(Component component, int index)
            throws IndexOutOfBoundsException {
        int componentCount = getComponentCount();

        if (index < 0 || index > componentCount) {
            throw new IndexOutOfBoundsException();
        }

        if (componentCount == 0 || index == componentCount) {
            addComponent(component);
            return;
        }

        if (component.getParent().isPresent()) {
            if (isChild(component)) {
                // When component is removed, all components after it are
                // shifted down
                if (index > getComponentIndex(component)) {
                    index--;
                }
            }
            AbstractSingleComponentContainer.removeFromParent(component);
        }
        if (component instanceof AbstractComponent) {
            ((AbstractComponent) component).setParent(null);
        }

        Element container = getContainer();
        if (isSpacing && getComponentCount() > 0) {
            index = index * 2;
            // adding to end has been already covered, so spacing is always next
            container.insertChild(index, wrapInSlot(component),
                    createSpacing());
        } else {
            container.insertChild(index, wrapInSlot(component));
        }

        setComponentAlignment(component, getDefaultComponentAlignment());
        addDetachListener(component);
        fireComponentAttachEvent(component);
        markAsDirtyRecursive();
    }

    /**
     * Adds a component into this container. The component is added to the left
     * or on top of the other components.
     *
     * @param component
     *            the component to be added.
     */
    public void addComponentAsFirst(Component component) {
        addComponent(component, 0);
    }

    @Override
    public void replaceComponent(Component oldComponent,
            Component newComponent) {

        if (!isChild(oldComponent)) {
            addComponent(newComponent);
            return;
        }

        // just swap the components
        if (isChild(newComponent)) {
            movingChildrenInternally = true;
            Element oldElement = oldComponent.getElement();
            Element newElement = newComponent.getElement();
            Element oldComponentSlot = oldElement.getParent();
            Element newComponentSlot = newElement.getParent();

            newComponentSlot.appendChild(oldElement);
            oldComponentSlot.appendChild(newElement);
            movingChildrenInternally = false;

            setSlotClasses();
            return;
        }

        addComponent(newComponent, getComponentIndex(oldComponent));
        setComponentAlignment(newComponent,
                getComponentAlignment(oldComponent));
        remove(oldComponent);
    }

    /**
     * Gets the number of contained components.
     *
     * @return the number of contained components
     */
    @Override
    public int getComponentCount() {
        return (int) getChildren().count();
    }

    /**
     * Returns the component at the given position.
     *
     * @param index
     *            The position of the component.
     * @return The component at the given index.
     * @throws IndexOutOfBoundsException
     *             If the index is out of range.
     */
    public Component getComponent(int index) throws IndexOutOfBoundsException {
        return getChildren().collect(Collectors.toList()).get(index);
    }

    /**
     * Returns the index of the given component.
     *
     * @param component
     *            The component to look up.
     * @return The index of the component or -1 if the component is not a child.
     */
    public int getComponentIndex(Component component) {
        return getChildren().collect(Collectors.toList()).indexOf(component);
    }

    protected void setSlotClasses() {
        getChildren().forEach(child -> {
            Element slot = child.getElement().getParent();

            // not adding styles for Flow components as it would be painful to
            // try to monitor the added/removed styles for those, and it is
            // unclear if it would be even necessary
            if (child instanceof AbstractComponent) {
                StringBuilder sb = new StringBuilder(SLOT_CLASS_NAME);
                ((AbstractComponent) child).getCustomStyles().forEach(style -> {
                    sb.append(" ").append(SLOT_CLASS_NAME).append("-")
                            .append(style);
                });
                slot.setAttribute("class", sb.toString());
            }

            ClassList classList = slot.getClassList();
            AlignmentInfo alignment = new AlignmentInfo(
                    ComponentUtil.getData(child, AlignmentKey.class).getValue()
                            .getBitMask());

            classList.set(ALIGN_CLASS_PREFIX + "center",
                    alignment.isHorizontalCenter());
            classList.set(ALIGN_CLASS_PREFIX + "right", alignment.isRight());

            classList.set(ALIGN_CLASS_PREFIX + "middle",
                    alignment.isVerticalCenter());
            classList.set(ALIGN_CLASS_PREFIX + "bottom", alignment.isBottom());
        });
    }

    @Override
    public void setComponentAlignment(Component childComponent,
            Alignment alignment) {
        if (!isChild(childComponent)) {
            throw new IllegalArgumentException(
                    "Component must be added to layout before using setComponentAlignment()");
        }

        ComponentUtil.setData(childComponent, AlignmentKey.class,
                new AlignmentKey(alignment));
        markAsDirty();
    }

    @Override
    public Alignment getComponentAlignment(Component childComponent) {
        if (!isChild(childComponent)) {
            throw new IllegalArgumentException(
                    "The given component is not a child of this layout");
        }

        return ComponentUtil.getData(childComponent, AlignmentKey.class)
                .getValue();
    }

    @Override
    public void setDefaultComponentAlignment(
            Alignment defaultComponentAlignment) {
        this.defaultComponentAlignment = defaultComponentAlignment;
    }

    @Override
    public Alignment getDefaultComponentAlignment() {
        return defaultComponentAlignment;
    }

    @Override
    public Registration addLayoutClickListener(
            LayoutEvents.LayoutClickListener listener) {
        return getEventBus().addListener(LayoutEvents.LayoutClickEvent.class,
                listener);
    }

    static class ExpandRatioKey {

        private final float value;

        public ExpandRatioKey(float value) {
            this.value = value;
        }
    }

    static class AlignmentKey {
        private final Alignment value;

        public AlignmentKey(Alignment value) {
            this.value = value;
        }

        public Alignment getValue() {
            return value;
        }
    }
}
