/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */
package com.vaadin.classic.v8.event;

import com.vaadin.classic.v8.shared.MouseEventDetails;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentEvent;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.component.DomEvent;
import com.vaadin.flow.component.EventData;
import com.vaadin.classic.v8.ui.Event;

/**
 * Legacy version of MouseEvents interface that serves as a wrapper for mouse
 * related events.
 *
 * @author Vaadin Ltd.
 * @see ClickListener
 */
public interface MouseEvents {

    /**
     * Legacy MouseEvent class for holding information about a mouse click
     * event. A {@link ClickEvent} is fired when the user clicks on a
     * <code>Component</code>.
     *
     * The information available for click events are terminal dependent.
     * Correct values for all event details cannot be guaranteed.
     *
     * @author Vaadin Ltd.
     * @see ClickListener
     */
    @DomEvent("mouseup")
    class ClickEvent extends Event {
        private MouseEventDetails details = new MouseEventDetails();

        public ClickEvent(Component source, boolean fromClient,
                @EventData("event.clientX") int clientX,
                @EventData("event.clientY") int clientY,
                @EventData("event.detail") int clickCount,
                @EventData("event.button") int button,
                @EventData("event.ctrlKey") boolean ctrlKey,
                @EventData("event.shiftKey") boolean shiftKey,
                @EventData("event.altKey") boolean altKey,
                @EventData("event.metaKey") boolean metaKey,
                @EventData("event.clientX - element.getBoundingClientRect().x") int relativeX,
                @EventData("event.clientY - element.getBoundingClientRect().y") int relativeY) {
            super(source, fromClient);
            details.setButton(getMouseButton(button));
            details.setClientX(clientX);
            details.setClientY(clientY);
            details.setCtrlKey(ctrlKey);
            details.setShiftKey(shiftKey);
            details.setAltKey(altKey);
            details.setMetaKey(metaKey);
            details.setRelativeX(relativeX);
            details.setRelativeY(relativeY);
            details.setType(clickCount == 2 ? 2 : 1);
        }

        /**
         * Constructs a server-side ClickEvent with the given mouseEventDetails
         */
        public ClickEvent(Component source,
                MouseEventDetails mouseEventDetails) {
            super(source, false);
            details = mouseEventDetails;
        }

        /**
         * Gets the component where the event occurred.
         *
         * @return the source component of the event
         */
        public Component getComponent() {
            return getSource();
        }

        /**
         * Returns an identifier describing which mouse button the user pushed.
         * Compare with {@link MouseEventDetails.MouseButton#LEFT},{@link MouseEventDetails.MouseButton#MIDDLE},
         * {@link MouseEventDetails.MouseButton#RIGHT} to find out which button it is.
         *
         * @return one of {@link MouseEventDetails.MouseButton#LEFT}, {@link MouseEventDetails.MouseButton#MIDDLE},
         *         {@link MouseEventDetails.MouseButton#RIGHT}.
         */
        public MouseEventDetails.MouseButton getButton() {
            return details.getButton();
        }

        /**
         * Returns the mouse position (x coordinate) when the click took place.
         * The position is relative to the browser client area.
         *
         * @return The mouse cursor x position
         */
        public int getClientX() {
            return details.getClientX();
        }

        /**
         * Returns the mouse position (y coordinate) when the click took place.
         * The position is relative to the browser client area.
         *
         * @return The mouse cursor y position
         */
        public int getClientY() {
            return details.getClientY();
        }

        /**
         * Returns the relative mouse position (x coordinate) when the click
         * took place. The position is relative to the clicked component.
         *
         * @return The mouse cursor x position relative to the clicked layout
         *         component or -1 if no x coordinate available
         */
        public int getRelativeX() {
            return details.getRelativeX();
        }

        /**
         * Returns the relative mouse position (y coordinate) when the click
         * took place. The position is relative to the clicked component.
         *
         * @return The mouse cursor y position relative to the clicked layout
         *         component or -1 if no y coordinate available
         */
        public int getRelativeY() {
            return details.getRelativeY();
        }

        /**
         * Checks if the event is a double click event.
         *
         * @return true if the event is a double click event, false otherwise
         */
        public boolean isDoubleClick() {
            return details.isDoubleClick();
        }

        /**
         * Checks if the Alt key was down when the mouse event took place.
         *
         * @return true if Alt was down when the event occurred, false otherwise
         */
        public boolean isAltKey() {
            return details.isAltKey();
        }

        /**
         * Checks if the Ctrl key was down when the mouse event took place.
         *
         * @return true if Ctrl was pressed when the event occurred, false
         *         otherwise
         */
        public boolean isCtrlKey() {
            return details.isCtrlKey();
        }

        /**
         * Checks if the Meta key was down when the mouse event took place.
         *
         * @return true if Meta was pressed when the event occurred, false
         *         otherwise
         */
        public boolean isMetaKey() {
            return details.isMetaKey();
        }

        /**
         * Checks if the Shift key was down when the mouse event took place.
         *
         * @return true if Shift was pressed when the event occurred, false
         *         otherwise
         */
        public boolean isShiftKey() {
            return details.isShiftKey();
        }

        /**
         * Returns a human readable string representing which button has been
         * pushed. This is meant for debug purposes only and the string returned
         * could change. Use {@link #getButton()} to check which button was
         * pressed.
         *
         * @return A string representation of which button was pushed.
         */
        public String getButtonName() {
            return details.getButtonName();
        }

        private static MouseEventDetails.MouseButton getMouseButton(int button) {
            if (button == 0)
                return MouseEventDetails.MouseButton.LEFT;
            if (button == 1)
                return MouseEventDetails.MouseButton.MIDDLE;
            if (button == 2)
                return MouseEventDetails.MouseButton.RIGHT;

            return null;
        }

        /**
         * Returns an information about mouse event like position, buttons
         * pressed etc.
         *
         * @return An information about mouse event
         */
        public MouseEventDetails getMouseEventDetails() {
            return details;
        }
    }

    /**
     * Interface for listening for a {@link ClickEvent} fired by a
     * {@link Component}.
     *
     * @author Vaadin Ltd.
     * @see ClickEvent
     */
    @FunctionalInterface
    public interface ClickListener extends ComponentEventListener<ClickEvent> {

        default void onComponentEvent(ClickEvent event) {
            this.click(event);
        }

        /**
         * Called when a {@link Component} has been clicked. A reference to the
         * component is given by {@link ClickEvent#getComponent()}.
         *
         * @param event
         *            An event containing information about the click.
         */
        public void click(ClickEvent event);
    }

    /**
     * Class for holding additional event information for DoubleClick events.
     * Fired when the user double-clicks on a <code>Component</code>.
     *
     * @author Vaadin Ltd.
     * @see ClickEvent
     */
    public static class DoubleClickEvent extends ComponentEvent<Component> {

        public DoubleClickEvent(Component source) {
            super(source, true);
        }

        /**
         * Gets the component where the event occurred.
         *
         * @return the source component of the event
         */
        public Component getComponent() {
            return getSource();
        }

    }

    /**
     * Legacy interface for listening for a {@link DoubleClickEvent} fired by a
     * {@link Component}.
     *
     * @author Vaadin Ltd.
     * @see DoubleClickEvent
     */
    @FunctionalInterface
    public interface DoubleClickListener
            extends ComponentEventListener<DoubleClickEvent> {

        @Override
        default void onComponentEvent(DoubleClickEvent event) {
            this.doubleClick(event);
        }

        /**
         * Called when a {@link Component} has been double clicked. A reference
         * to the component is given by {@link DoubleClickEvent#getComponent()}.
         *
         * @param event
         *            An event containing information about the double click.
         */
        public void doubleClick(DoubleClickEvent event);
    }

}
