/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.stream.Collectors;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.shared.Registration;

/**
 * Legacy version of AbstractComponentContainer that resembles Vaadin 7/8's
 * AbstractComponentContainer API as closely as possible in order to facilitate
 * migration to newer versions of Vaadin.
 * <p>
 * Extension to {@link AbstractComponent} that defines the default
 * implementation for the methods in {@link ComponentContainer}. Basic UI
 * components that need to contain other components inherit this class to easily
 * qualify as a component container.
 *
 * @author Vaadin Ltd
 */
public abstract class AbstractComponentContainer extends AbstractComponent
        implements ComponentContainer {

    @Override
    public void addComponent(Component component) {
        if (component.getParent().isPresent()) {
            // If the component already has a parent, try to remove it
            AbstractSingleComponentContainer.removeFromParent(component);
        }
        if (component instanceof AbstractComponent) {
            // make sure old parent is marked dirty (if exists)
            ((AbstractComponent) component).setParent(null);
        }
        ComponentContainer.super.add(component);
        fireComponentAttachEvent(component);
        markAsDirtyRecursive();
    }

    @Override
    public void addComponents(Component... components) {
        for (Component component : components) {
            addComponent(component);
        }
    }

    @Override
    public void add(Component... components) {
        // legacy framework does the addition one at a time, so does this
        for (Component component : components) {
            addComponent(component);
        }
    }

    /**
     * {@inheritDoc}
     * <p>
     * Note that only {@link Component components} are moved from
     * <code>source</code> into this container. If an element in the
     * <code>source</code> does not have a component, it will remain in the
     * <code>source</code>.
     */
    @Override
    public void moveComponentsFrom(ComponentContainer source) {
        final ArrayList<Component> componentsToMove = new ArrayList<>(
                getComponentCount());
        source.iterator().forEachRemaining(componentsToMove::add);

        for (Component component : componentsToMove) {
            source.removeComponent(component); // will mark it as dirty
        }

        addComponents(componentsToMove.toArray(new Component[0]));
    }

    @Override
    public void removeAllComponents() {
        removeAll();
    }

    @Override
    public void removeAll() {
        // collect due to removal
        getChildren().collect(Collectors.toList())
                .forEach(this::removeComponent);
    }

    @Override
    public void removeComponent(Component component) {
        if (equals(component.getParent().orElse(null))) {
            ComponentContainer.super.remove(component);
            fireComponentDetachEvent(component);
            markAsDirtyRecursive();
        }
    }

    @Override
    public void remove(Component... components) {
        for (Component component : components) {
            removeComponent(component);
        }
    }

    /**
     * Gets the component container iterator for going through all the
     * components in the container.
     *
     * @return the Iterator of the components inside the container.
     */
    @Override
    public Iterator<Component> iterator() {
        return getChildren().iterator();
    }

    @Override
    public void setHeight(float height, Unit unit) {
        if (height != getHeight() || unit != getHeightUnits()) {
            super.setHeight(height, unit);
            // legacy framework goes through all child-tree to determine which
            // children need repaint, but since there is no "repaint" that costs
            // too much to do on the client side, removing the optimization as
            // unnecessary
            markAsDirtyRecursive();
        }
    }

    @Override
    public void setWidth(float width, Unit unit) {
        if (width != getWidth() || unit != getWidthUnits()) {
            super.setWidth(width, unit);
            // legacy framework goes through all child-tree to determine which
            // children need repaint, but since there is no "repaint" that costs
            // too much to do on the client side, removing the optimization as
            // unnecessary
            markAsDirtyRecursive();
        }
    }

    @Override
    public Registration addComponentAttachListener(
            ComponentAttachListener listener) {
        return getEventBus().addListener(ComponentAttachEvent.class, listener);
    }

    @Override
    public Registration addComponentDetachListener(
            ComponentDetachListener listener) {
        return getEventBus().addListener(ComponentDetachEvent.class, listener);
    }

    /**
     * Fires the component attached event. This should be called by the
     * addComponent methods after the component have been added to this
     * container.
     *
     * @param component
     *            the component that has been added to this container.
     */
    protected void fireComponentAttachEvent(Component component) {
        getEventBus().fireEvent(new ComponentAttachEvent(this, component));
    }

    /**
     * Fires the component detached event. This should be called by the
     * removeComponent methods after the component have been removed from this
     * container.
     *
     * @param component
     *            the component that has been removed from this container.
     */
    protected void fireComponentDetachEvent(Component component) {
        getEventBus().fireEvent(new ComponentDetachEvent(this, component));
    }
}
