/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicReference;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.Registration;

/**
 * Abstract base class for legacy Vaadin framework version 7/8 component
 * containers that have only one child component.
 * <p>
 * For component containers that support multiple children, inherit
 * {@link AbstractComponentContainer} instead of this class.
 */
public abstract class AbstractSingleComponentContainer extends AbstractComponent
        implements SingleComponentContainer {
    private Element wrapper;
    private boolean internalRemoval;

    @Override
    public int getComponentCount() {
        return (getContent() != null) ? 1 : 0;
    }

    @Override
    public Iterator<Component> iterator() {
        if (getContent() != null) {
            return Collections.singletonList(getContent()).iterator();
        } else {
            return Collections.emptyIterator();
        }
    }

    @Override
    public Registration addComponentAttachListener(
            ComponentAttachListener listener) {
        return getEventBus().addListener(ComponentAttachEvent.class, listener);
    }

    @Override
    public Registration addComponentDetachListener(
            ComponentDetachListener listener) {
        return getEventBus().addListener(ComponentDetachEvent.class, listener);
    }

    /**
     * Fires the component attached event. This is called by the
     * {@link #setContent(Component)} method after the component has been set as
     * the content.
     *
     * @param component
     *            the component that has been added to this container.
     */
    protected void fireComponentAttachEvent(Component component) {
        fireEvent(new ComponentAttachEvent(this, component));
    }

    /**
     * Fires the component detached event. This is called by the
     * {@link #setContent(Component)} method after the content component has
     * been replaced by other content.
     *
     * @param component
     *            the component that has been removed from this container.
     */
    protected void fireComponentDetachEvent(Component component) {
        fireEvent(new ComponentDetachEvent(this, component));
    }

    @Override
    public Component getContent() {
        return getContentWrapper().getChildren().findFirst()
                .flatMap(Element::getComponent).orElse(null);
    }

    /**
     * Sets the content of this container. The content is a component that
     * serves as the outermost item of the visual contents.
     * <p>
     * The content must always be set, either with a constructor parameter or by
     * calling this method.
     * <p>
     * Previous versions of Vaadin used a {@link VerticalLayout} with margins
     * enabled as the default content but that is no longer the case.
     *
     * @param content
     *            a component (typically a layout) to use as content
     */
    @Override
    public void setContent(Component content) {
        if (isOrHasAncestor(content)) {
            throw new IllegalArgumentException(
                    "Component cannot be added inside it's own content");
        }

        Component oldContent = getContent();
        if (oldContent == content) {
            // do not set the same content twice
            return;
        }

        if (oldContent != null && equals(oldContent.getParent().orElse(null))) {
            if (getContentWrapper()
                    .indexOfChild(oldContent.getElement()) != -1) {
                internalRemoval = true;
                getContentWrapper().removeChild(oldContent.getElement());
                internalRemoval = false;
            }
            fireComponentDetachEvent(oldContent);
        }

        if (content != null) {
            AbstractSingleComponentContainer.removeFromParent(content);
            getContentWrapper().appendChild(content.getElement());
            fireComponentAttachEvent(content);
            if (isAttached()) {
                addDetachListener(content);
            }
        }

        markAsDirtyRecursive();
    }

    /**
     * Sets the element that will be used as the content wrapper.
     *
     * <p>
     * By default, the content component defined through
     * {@link #setContent(Component)} is attached as a direct child of
     * {@link AbstractSingleComponentContainer}. That's sometimes not the
     * desired behavior if, for instance, the container wants to define a
     * different slot on which the content will be attached to.
     * </p>
     * <p>
     * Note that it might have a side effect where {@link #getParent()} from the
     * content component returns the wrapper instead of the
     * {@link AbstractSingleComponentContainer} instance. To avoid this, the
     * {@link Element} API can be used to create the element and
     * {@link com.vaadin.flow.component.ComponentUtil#componentFromElement(Element, Class, boolean)}
     * to get the component passing <code>false</code> to the
     * <code>mapComponent</code> property, so it won't map the component to the
     * element. Or use the {@link com.vaadin.flow.dom.ElementFactory} methods.
     * </p>
     *
     * <p>
     * The wrapper can only be set once and if no content is already set to the
     * container.
     * </p>
     *
     * @param wrapper
     *            the wrapper element to be used to attach the content to
     */
    protected void setContentWrapper(Element wrapper) {
        if (this.wrapper != null) {
            throw new UnsupportedOperationException(
                    "Wrapper element can only be set once.");
        }
        if (getContent() != null) {
            throw new UnsupportedOperationException(
                    "Wrapper element can only be set if no content is defined");
        }
        assert wrapper != null : "Wrapper should be not null";

        this.wrapper = wrapper;
        getElement().appendChild(wrapper);
    }

    /**
     * Get the wrapper to be used as parent of the content element. If no
     * wrapper is set, <code>this.getElement()</code> is returned.
     *
     * @return the wrapper or <code>this.getElement()</code> element.
     */
    protected Element getContentWrapper() {
        return wrapper != null ? wrapper : getElement();
    }

    @Override
    protected void onAttach(com.vaadin.flow.component.AttachEvent attachEvent) {
        super.onAttach(attachEvent);
        if (getContent() != null) {
            addDetachListener(getContent());
        }
    }

    private void addDetachListener(Component component) {
        AtomicReference<Registration> holder = new AtomicReference<>();
        // Only fire the component detach event when the component
        // is actually removed.
        holder.set(component.getElement().addDetachListener(event -> {
            // FIXME #118 If the panel component is moved, this event will be
            // fired unnecessarily as there is no clear way to fix it at the
            // moment. It would be easy to fix if the DetachEvent from Flow
            // would have metadata that what was actually detached.
            if (!internalRemoval) {
                fireComponentDetachEvent(component);
                markAsDirty();
            }
            holder.get().remove();
        }));
    }

    /**
     * Utility method for removing a component from its parent (if possible).
     *
     * @param content
     *            component to remove
     */
    public static void removeFromParent(Component content)
            throws IllegalArgumentException {
        // Verify the appropriate session is locked
        UI parentUI = content.getUI().orElse(null);
        if (parentUI != null) {
            VaadinSession parentSession = parentUI.getSession();
            if (parentSession != null && !parentSession.hasLock()) {
                String message = "Cannot remove from parent when the session is not locked.";
                if (VaadinService.isOtherSessionLocked(parentSession)) {
                    message += " Furthermore, there is another locked session, indicating that the component might be about to be moved from one session to another.";
                }
                throw new IllegalStateException(message);
            }
        }

        Component parent = content.getParent().orElse(null);
        // only do this check/action for legacy components to get
        // ComponentDetachEvent fired correctly
        if (parent instanceof AbstractComponent) {
            if (parent instanceof ComponentContainer) {
                // If the component already has a parent, try to remove it
                ComponentContainer oldParent = (ComponentContainer) parent;
                oldParent.removeComponent(content);
            } else if (parent instanceof SingleComponentContainer) {
                SingleComponentContainer oldParent = (SingleComponentContainer) parent;
                if (oldParent.getContent() == content) {
                    oldParent.setContent(null);
                }
            } else if (parent != null) {
                throw new IllegalArgumentException(
                        "Content is already attached to another parent");
            }
        }
    }

    @Override
    public void setHeight(float height, Unit unit) {
        if (height != getHeight() || unit != getHeightUnits()) {
            super.setHeight(height, unit);
            // legacy framework goes through all child-tree to determine which
            // children need repaint, but since there is no "repaint" that costs
            // too much to do on the client side, removing the optimization as
            // unnecessary
            markAsDirtyRecursive();
        }
    }

    @Override
    public void setWidth(float width, Unit unit) {
        if (width != getWidth() || unit != getWidthUnits()) {
            super.setWidth(width, unit);
            // legacy framework goes through all child-tree to determine which
            // children need repaint, but since there is no "repaint" that costs
            // too much to do on the client side, removing the optimization as
            // unnecessary
            markAsDirtyRecursive();
        }
    }
}
