/*
 * Copyright (c) 2000-2022 Vaadin Ltd.
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
 */

package com.vaadin.classic.v8.ui;

import com.vaadin.classic.v8.server.Helpers;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.Unit;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.dom.ElementFactory;
import com.vaadin.classic.v8.shared.ui.ContentMode;
import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;

import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Legacy version of Label component for showing non-editable short texts.
 * <p>
 * The label content can be set to the modes specified by {@link ContentMode}.
 * If content mode is set to HTML, any HTML content is allowed.
 *
 * @author Vaadin Ltd.
 */
@Tag(Tag.DIV)
@CssImport("./label.css")
@JsModule("@vaadin/vaadin-lumo-styles/font-icons.js") // necessary for icons
                                                      // used in ValoTheme, e.g.
                                                      // ValoTheme.LABEL_SUCCESS
public class Label extends AbstractComponent {
    private static final String V_CAPTION = "v-caption";
    private static final String V_CAPTIONTEXT = "v-captiontext";
    private static final String V_HAS_CAPTION = "v-has-caption";
    private static final String V_CAPTION_ON_TOP = "v-caption-on-top";
    private static long idCounter = 0;

    private String caption;
    private Element vCaption;
    private Element vCaptionText;
    private final VLabel vLabel;
    private final String captionId;
    private boolean isCaptionAsHtml;
    private boolean isLabelUsingServerGeneratedId;

    @Override
    public void beforeClientResponse(boolean initial) {
        // No need to call super.beforeClientResponse, we only need the internal
        // style classes on the wrapper.
        // width, height and style names will be added to the inner vLabel
        // component
        getElement().setAttribute("class",
                String.join(" ", getInternalStyles()));

        if (vCaption != null) {
            vCaption.setAttribute("class", createVCaptionClassNames());
        }
    }

    /**
     * Creates an empty Label.
     */
    public Label() {
        this("");
    }

    /**
     * Creates a new instance with text content mode and the given text.
     *
     * @param text
     *            the text to set
     */
    public Label(String text) {
        this(text, ContentMode.TEXT);
    }

    /**
     * Creates a new instance with the given text and content mode.
     *
     * @param text
     *            the text to set
     * @param contentMode
     *            the content mode to use
     */
    public Label(String text, ContentMode contentMode) {
        // getElement().getStyle().set("display", "contents !important"); can't
        // use !important due to https://github.com/vaadin/flow/issues/11981
        getElement().getStyle().set("display", "contents");

        captionId = generateId();
        vLabel = new VLabel(text, contentMode);
        getElement().appendChild(vLabel.getElement());

        if (Helpers.isVaadin7Defaults()) {
            setWidth(100, Unit.PERCENTAGE);
        }
    }

    /**
     * Gets the caption of the component.
     *
     * @return the caption of the component or {@code null} if the caption is
     *         not set.
     * @see #setCaption(String)
     */
    public String getCaption() {
        return caption;
    }

    /**
     * Sets the caption of the component.
     *
     * @param caption
     *            the new caption for the component. If the caption is
     *            {@code null}, no caption is shown
     */
    public void setCaption(String caption) {
        if (Objects.equals(caption, this.caption)) {
            return;
        }

        this.caption = caption;

        if (this.caption == null) {
            removeCaption();
        } else {
            addCaption();
        }

        markAsDirty();
    }

    /**
     * Gets the content mode of the label.
     *
     * @return the content mode of the label
     * @see ContentMode
     */
    public ContentMode getContentMode() {
        return vLabel.getContentMode();
    }

    /**
     * Sets the content mode of the label.
     *
     * @param contentMode
     *            the content mode to set
     * @see ContentMode
     */
    public void setContentMode(ContentMode contentMode) {
        vLabel.setContentMode(contentMode);
    }

    /**
     * Sets the text to be shown in the label.
     *
     * @param value
     *            the text to show in the label, null is converted to an empty
     *            string
     */
    public void setValue(String value) {
        vLabel.setValue(value);
    }

    /**
     * Gets the text shown in the label.
     *
     * @return the text shown in the label, not null
     */
    public String getValue() {
        return vLabel.getValue();
    }

    /**
     * Sets whether the caption is rendered as HTML.
     * <p>
     * If set to true, the captions are rendered in the browser as HTML and the
     * developer is responsible for ensuring no harmful HTML is used. If set to
     * false, the caption is rendered in the browser as plain text.
     * <p>
     * Note: Unlike the Label component in Vaadin 7/8, the script tag is removed
     * when the caption is rendered as html
     *
     * The default is false, i.e. to render that caption as plain text.
     *
     * @param captionAsHtml
     *            true if the captions are rendered as HTML, false if rendered
     *            as plain text
     */
    public void setCaptionAsHtml(boolean captionAsHtml) {
        isCaptionAsHtml = captionAsHtml;
        updateVCaptionText();
    }

    /**
     * Checks whether captions are rendered as HTML
     * <p>
     * The default is false, i.e. to render that caption as plain text.
     *
     * @return true if the captions are rendered as HTML, false if rendered as
     *         plain text
     */
    public boolean isCaptionAsHtml() {
        return isCaptionAsHtml;
    }

    @Override
    public Stream<Component> getChildren() {
        return Stream.empty();
    }

    @Override
    public void setWidth(String width) {
        vLabel.setWidth(width);
    }

    @Override
    public void setWidth(float width, Unit unit) {
        vLabel.setWidth(width, unit);
    }

    @Override
    public float getWidth() {
        return vLabel.getWidth();
    }

    @Override
    public Unit getWidthUnits() {
        return vLabel.getWidthUnits();
    }

    @Override
    public void setHeight(String height) {
        vLabel.setHeight(height);
    }

    @Override
    public void setHeight(float height, Unit unit) {
        vLabel.setHeight(height, unit);
    }

    @Override
    public float getHeight() {
        return vLabel.getHeight();
    }

    @Override
    public Unit getHeightUnits() {
        return vLabel.getHeightUnits();
    }

    @Override
    public void setSizeFull() {
        vLabel.setSizeFull();
    }

    @Override
    public void setWidthFull() {
        vLabel.setWidthFull();
    }

    @Override
    public void setHeightFull() {
        vLabel.setHeightFull();
    }

    @Override
    public void setSizeUndefined() {
        vLabel.setSizeUndefined();
    }

    @Override
    public void setWidthUndefined() {
        vLabel.setWidthUndefined();
    }

    @Override
    public void setHeightUndefined() {
        vLabel.setHeightUndefined();
    }

    @Override
    public String getStyleName() {
        return vLabel.getStyleName();
    }

    @Override
    public void setStyleName(String style) {
        vLabel.setStyleName(style);
    }

    @Override
    public void setStyleName(String style, boolean add) {
        vLabel.setStyleName(style, add);
    }

    @Override
    public void addStyleName(String style) {
        vLabel.addStyleName(style);
    }

    @Override
    public void addStyleNames(String... styles) {
        vLabel.addStyleNames(styles);
    }

    @Override
    public void removeStyleName(String style) {
        vLabel.removeStyleName(style);
    }

    @Override
    public void removeStyleNames(String... styles) {
        vLabel.removeStyleNames(styles);
    }

    @Override
    public String getPrimaryStyleName() {
        return vLabel.getPrimaryStyleName();
    }

    @Override
    public void setPrimaryStyleName(String style) {
        vLabel.setPrimaryStyleName(style);
    }

    @Override
    protected Collection<String> getCustomStyles() {
        return vLabel.getCustomStyles();
    }

    @Override
    public boolean isEnabled() {
        return vLabel.isEnabled();
    }

    @Override
    public void setEnabled(boolean enabled) {
        markAsDirty(); // vCaption classnames needs to be updated
        vLabel.setEnabled(enabled);
    }

    @Override
    public void setId(String id) {
        // if we generated the id, don't remove it
        if (isLabelUsingServerGeneratedId && "".equals(id)) {
            return;
        }

        vLabel.setId(id);
        isLabelUsingServerGeneratedId = false;
        setCaptionHtmlForAttribute();
    }

    @Override
    public Optional<String> getId() {
        return vLabel.getId();
    }

    private String createVCaptionClassNames() {
        Stream<String> vCaptionClasses = Stream.concat(
                Stream.of(V_CAPTION, isEnabled() ? "" : "v-disabled"),
                getCustomStyles().stream()
                        .map(styleClass -> V_CAPTION + "-" + styleClass));

        return String.join(" ", vCaptionClasses.collect(Collectors.toSet()));
    }

    private void removeCaption() {
        // getElement().getStyle().set("display", "contents !important"); can't
        // use !important due to https://github.com/vaadin/flow/issues/11981
        getElement().getStyle().set("display", "contents");
        getElement().removeChild(vCaption);
        vCaption = null;
        vCaptionText = null;
        removeInternalStyles(V_HAS_CAPTION, V_CAPTION_ON_TOP);
        vLabel.removeAriaLabelledBy();
    }

    private void addCaption() {
        if (vCaption != null) {
            updateVCaptionText();
            return;
        }

        getElement().getStyle().remove("display");
        addCaptionElements();
        setCaptionHtmlForAttribute();
        addInternalStyles(V_HAS_CAPTION, V_CAPTION_ON_TOP);
        vLabel.setAriaLabelledBy(captionId);
    }

    private void updateVCaptionText() {
        if (caption == null) {
            return;
        }
        // FW behavior
        String text = caption.isEmpty() ? " " : caption;

        if (isCaptionAsHtml()) {
            // Unlike FW, we're removing script tags
            vCaptionText.setProperty("innerHTML",
                    Jsoup.clean(text, Safelist.relaxed()));
        } else {
            vCaptionText.setText(text);
        }
    }

    private void addCaptionElements() {
        if (vCaption != null) {
            return;
        }

        vCaptionText = ElementFactory.createSpan();
        vCaptionText.getClassList().add(V_CAPTIONTEXT);
        updateVCaptionText();

        vCaption = ElementFactory.createDiv();
        vCaption.setAttribute("id", captionId);
        vCaption.appendChild(vCaptionText);

        getElement().insertChild(0, vCaption);
    }

    private void setCaptionHtmlForAttribute() {
        if (vCaption == null) {
            return;
        }

        // For a11y purposes, the 'for' attribute of the caption should refer to
        // the vLabel's id,
        // if we don't already have an 'id', we have to generate a random id and
        // use that
        // to connect vCaption<->vLabel
        if (!getId().isPresent()) {
            vLabel.setId(generateId());
            isLabelUsingServerGeneratedId = true;
        }
        vCaption.setAttribute("for", getId().get());
    }

    private String generateId() {
        return "lcp-label-" + (++Label.idCounter);
    }

    @Tag(Tag.DIV)
    static class VLabel extends AbstractComponent {
        private static final String defaultPrimaryStyle = "v-label";
        private static final String V_LABEL_UNDEF_W = "v-label-undef-w";

        private ContentMode contentMode;
        private String value;

        @Override
        public void beforeClientResponse(boolean initial) {
            super.beforeClientResponse(initial);
            String primaryStyleName = getPrimaryStyleName();

            /*
             * Not an exact one-to-one map to FW. FW has two bugs in regard to
             * *-undef-w class.
             *
             * if you do: 1. set primary Style 2. set width FW doesn't remove
             * primary-undef-w although the component has a defined width
             *
             * if you do: 1. a. set primary style b. set width 2. remove width
             * FW doesn't add primary-undef-w although the component doesn't
             * have a defined width
             *
             * I don't think anyone would rely on these behaviors, so it is not
             * replicated here
             */
            if (primaryStyleName != null
                    && !defaultPrimaryStyle.equals(primaryStyleName)) {
                getElement().getClassList().set(primaryStyleName + "-undef-w",
                        getWidth() == SIZE_UNDEFINED);
            }
            getElement().getClassList().set(V_LABEL_UNDEF_W,
                    getWidth() == SIZE_UNDEFINED);
        }

        public VLabel(String text, ContentMode contentMode) {
            setPrimaryStyleName(defaultPrimaryStyle);
            value = text;
            this.contentMode = contentMode;

            updatePresentation();
        }

        public ContentMode getContentMode() {
            return contentMode;
        }

        public void setContentMode(ContentMode contentMode) {
            if (contentMode == this.contentMode) {
                return;
            }

            this.contentMode = contentMode;
            updatePresentation();
        }

        public void setValue(String value) {
            this.value = value == null ? "" : value;
            updatePresentation();
        }

        public String getValue() {
            return value;
        }

        public void removeAriaLabelledBy() {
            getElement().removeAttribute("aria-labelledby");
        }

        public void setAriaLabelledBy(String id) {
            getElement().setAttribute("aria-labelledby", id);
        }

        @Override
        protected void notifyParentAboutStyleChanges() {
            getParent().ifPresent(parent -> {
                ((Label) parent).markAsDirty(); // caption needs to receive the
                                                // styles
                ((Label) parent).notifyParentAboutStyleChanges(); // need to
                                                                  // update slot
                                                                  // styles for
                                                                  // HL+VL
            });
        }

        private void updatePresentation() {
            switch (contentMode) {
            case TEXT:
                getElement().setText(value);
                break;
            case PREFORMATTED:
                getElement().removeAllChildren();
                getElement()
                        .appendChild(ElementFactory.createPreformatted(value));
                break;
            case HTML:
                // Unlike FW, we're removing script tags
                getElement().setProperty("innerHTML",
                        Jsoup.clean(value, Safelist.relaxed()));
                break;
            }
        }
    }
}
