/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.client.data;

import java.util.function.Consumer;

import com.google.gwt.core.client.Scheduler;
import com.vaadin.shared.Range;

/**
 * Helper class for creating a {@link DataChangeHandler} for a Widget that does
 * not support lazy loading.
 *
 * @author Vaadin Ltd
 * @since 8.0
 */
public class SimpleDataChangeHandler implements DataChangeHandler {

    /**
     * Class to request the data source to get the full data set.
     */
    private static class DelayedResetScheduler {

        private final DataSource<?> dataSource;
        private boolean scheduled = false;

        public DelayedResetScheduler(DataSource<?> dataSource) {
            this.dataSource = dataSource;
        }

        public void schedule() {
            if (scheduled) {
                return;
            }
            Scheduler.get().scheduleFinally(() -> {
                dataSource.ensureAvailability(0, dataSource.size());
                scheduled = false;
            });
            scheduled = true;
        }

        public int getExpectedSize() {
            return dataSource.size();
        }

        public boolean isScheduled() {
            return scheduled;
        }
    }

    private final DelayedResetScheduler scheduler;
    private final Consumer<Range> refreshMethod;

    SimpleDataChangeHandler(DataSource<?> dataSource,
            Consumer<Range> refreshMethod) {
        scheduler = new DelayedResetScheduler(dataSource);
        this.refreshMethod = refreshMethod;
    }

    @Override
    public void dataUpdated(int firstRowIndex, int numberOfRows) {
        scheduler.schedule();
    }

    @Override
    public void dataRemoved(int firstRowIndex, int numberOfRows) {
        scheduler.schedule();
    }

    @Override
    public void dataAdded(int firstRowIndex, int numberOfRows) {
        scheduler.schedule();
    }

    @Override
    public void dataAvailable(int firstRowIndex, int numberOfRows) {
        if (!scheduler.isScheduled() && firstRowIndex == 0
                && numberOfRows == scheduler.getExpectedSize()) {
            // All data should now be available.
            refreshMethod.accept(Range.withLength(firstRowIndex, numberOfRows));
        } else {
            scheduler.schedule();
        }
    }

    @Override
    public void resetDataAndSize(int newSize) {
        scheduler.schedule();
    }
}
