/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.client.ui;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import com.google.gwt.dom.client.Element;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.HTML;
import com.vaadin.client.ApplicationConnection;
import com.vaadin.client.BrowserInfo;
import com.vaadin.client.UIDL;
import com.vaadin.client.Util;

/**
 * A widget class for the Embedded component.
 *
 * @author Vaadin Ltd
 */
public class VEmbedded extends HTML {
    /** The default classname for this widget. */
    public static String CLASSNAME = "v-embedded";

    /** For internal use only. May be removed or replaced in the future. */
    public Element browserElement;

    /** For internal use only. May be removed or replaced in the future. */
    public String type;

    /** For internal use only. May be removed or replaced in the future. */
    public String mimetype;

    /** For internal use only. May be removed or replaced in the future. */
    public ApplicationConnection client;

    /**
     * Constructs a widget for an Embedded component.
     */
    public VEmbedded() {
        setStyleName(CLASSNAME);
    }

    /**
     * Returns a map (name -> value) of all parameters in the UIDL.
     * <p>
     * For internal use only. May be removed or replaced in the future.
     *
     * @param uidl
     *            the UIDL to map
     * @return the parameter map
     */
    public static Map<String, String> getParameters(UIDL uidl) {
        Map<String, String> parameters = new HashMap<>();

        for (Object child : uidl) {
            if (child instanceof UIDL) {

                UIDL childUIDL = (UIDL) child;
                if (childUIDL.getTag().equals("embeddedparam")) {
                    String name = childUIDL.getStringAttribute("name");
                    String value = childUIDL.getStringAttribute("value");
                    parameters.put(name, value);
                }
            }

        }

        return parameters;
    }

    /**
     * Helper to return translated src-attribute from embedded's UIDL
     * <p>
     * For internal use only. May be removed or replaced in the future.
     *
     * @param src
     *            the src attribute
     * @param client
     *            the communication engine for this UI
     * @return the translated src-attribute or an empty String if not found
     */
    public String getSrc(String src, ApplicationConnection client) {
        String url = client.translateVaadinUri(src);
        if (url == null) {
            return "";
        }
        return url;
    }

    @Override
    protected void onDetach() {
        if (BrowserInfo.get().isIE()) {
            // Force browser to fire unload event when component is detached
            // from the view (IE doesn't do this automatically)
            if (browserElement != null) {
                /*
                 * src was previously set to javascript:false, but this was not
                 * enough to overcome a bug when detaching an iframe with a pdf
                 * loaded in IE9. about:blank seems to cause the adobe reader
                 * plugin to unload properly before the iframe is removed. See
                 * https://dev.vaadin.com/ticket/7855
                 */
                browserElement.setAttribute("src", "about:blank");
            }
        }
        super.onDetach();
    }

    @SuppressWarnings("deprecation")
    @Override
    public void onBrowserEvent(Event event) {
        super.onBrowserEvent(event);
        if (DOM.eventGetType(event) == Event.ONLOAD) {
            getLogger().info("Embeddable onload");
            Util.notifyParentOfSizeChange(this, true);
        }
    }

    private static Logger getLogger() {
        return Logger.getLogger(VEmbedded.class.getName());
    }
}
