/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.client.ui;

import com.vaadin.client.ui.VAbstractCalendarPanel.SubmitListener;

/**
 * A client side implementation for inline date field.
 *
 * @param <PANEL>
 *            the calendar panel type this field uses
 * @param <R>
 *            the resolution type which this field is based on (day, month, ...)
 * @author Vaadin Ltd
 */
public abstract class VAbstractDateFieldCalendar<PANEL extends VAbstractCalendarPanel<R>, R extends Enum<R>>
        extends VDateField<R> {

    /** For internal use only. May be removed or replaced in the future. */
    public final PANEL calendarPanel;

    /**
     * Constructs a date selection widget with an inline date/time selector.
     *
     * @param panel
     *            the calendar panel instance that should be displayed
     * @param resolution
     *            the resolution this widget should display (day, month, ...)
     */
    public VAbstractDateFieldCalendar(PANEL panel, R resolution) {
        super(resolution);
        calendarPanel = panel;
        calendarPanel.setParentField(this);
        add(calendarPanel);
        calendarPanel.setSubmitListener(new SubmitListener() {
            @Override
            public void onSubmit() {
                updateValueFromPanel();
            }

            @Override
            public void onCancel() {
                // NOP
            }
        });
        calendarPanel.setFocusOutListener(event -> {
            updateValueFromPanel();
            return false;
        });
    }

    /**
     * Update buffered values and send them (if any) to the server.
     */
    public abstract void updateValueFromPanel();

    /**
     * Sets the tabulator index for the calendar panel element that represents
     * the entire widget in the browser's focus cycle.
     *
     * @param tabIndex
     *            the new tabulator index
     */
    public void setTabIndex(int tabIndex) {
        calendarPanel.getElement().setTabIndex(tabIndex);
    }

    /**
     * Returns the tabulator index of the calendar panel element that represents
     * the entire widget in the browser's focus cycle.
     *
     * @return the tabulator index
     */
    public int getTabIndex() {
        return calendarPanel.getElement().getTabIndex();
    }
}
