/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.client.ui;

import java.util.Objects;

import com.google.gwt.user.client.ui.ListBox;
import com.vaadin.client.widgets.FocusableFlowPanelComposite;
import com.vaadin.shared.ui.nativeselect.NativeSelectState;

/**
 * The client-side widget for the {@code NativeSelect} component.
 *
 * @author Vaadin Ltd.
 */
public class VNativeSelect extends FocusableFlowPanelComposite {

    private final ListBox listBox = new ListBox();
    private boolean emptySelectionAllowed = true;

    /**
     * Creates a new {@code VNativeSelect} instance.
     */
    public VNativeSelect() {
        setStyleName(NativeSelectState.STYLE_NAME);
        getListBox().setStyleName(NativeSelectState.STYLE_NAME + "-select");
        getWidget().add(listBox);
    }

    @Override
    public void setStylePrimaryName(String style) {
        super.setStylePrimaryName(style);
        getListBox().setStyleName(style + "-select");
    }

    /**
     * Sets the selected item by its value. If given {@code null}, removes
     * selection.
     *
     * @param value
     *            the value of the item to select or {@code null} to select
     *            nothing
     */
    public void setSelectedItem(String value) {
        if (value == null) {
            if (emptySelectionAllowed) {
                getListBox().setSelectedIndex(0);
            } else {
                getListBox().setSelectedIndex(-1);
            }
        } else {
            for (int i = 0; i < getListBox().getItemCount(); i++) {
                if (Objects.equals(value, getListBox().getValue(i))) {
                    getListBox().setSelectedIndex(i);
                    break;
                }
            }
        }
    }

    /**
     * Sets the tab index.
     *
     * @param tabIndex
     *            the tab index to set
     */
    public void setTabIndex(int tabIndex) {
        getListBox().setTabIndex(tabIndex);
    }

    /**
     * Gets the underlying ListBox widget that this widget wraps.
     *
     * @return the ListBox this widget wraps
     */
    public ListBox getListBox() {
        return listBox;
    }

    @Override
    public void setWidth(String width) {
        if ("".equals(width)) {
            // undefined width
            getListBox().setWidth("");
        } else {
            // fill the composite
            getListBox().setWidth("100%");
        }
        super.setWidth(width);
    }

    @Override
    public void setHeight(String height) {
        if ("".equals(height)) {
            // undefined height
            getListBox().setHeight("");
        } else {
            // fill the composite
            getListBox().setHeight("100%");
        }
        super.setHeight(height);
    }

    /**
     * Sets the number of items that are visible. If only one item is visible,
     * then the box will be displayed as a drop-down list (the default).
     *
     * @since 8.1
     * @param visibleItemCount
     *            the visible item count
     */
    public void setVisibleItemCount(int visibleItemCount) {
        getListBox().setVisibleItemCount(visibleItemCount);
    }

    /**
     * Gets the number of items that are visible. If only one item is visible,
     * then the box will be displayed as a drop-down list.
     *
     * @since 8.1
     * @return the visible item count
     */
    public int getVisibleItemCount() {
        return getListBox().getVisibleItemCount();
    }

    /**
     * Returns true if empty selection is allowed.
     *
     * @since 8.7
     * @return empty selection is allowed
     */
    public boolean isEmptySelectionAllowed() {
        return emptySelectionAllowed;
    }

    /**
     * Sets true if empty selection is allowed.
     *
     * @since 8.7
     * @param emptySelectionAllowed
     */
    public void setEmptySelectionAllowed(boolean emptySelectionAllowed) {
        this.emptySelectionAllowed = emptySelectionAllowed;
    }
}
