/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.renderers;

import com.vaadin.shared.ui.grid.renderers.ComponentRendererState;
import com.vaadin.ui.AbstractField;
import com.vaadin.ui.Component;
import com.vaadin.ui.Grid;
import com.vaadin.ui.Label;

import elemental.json.Json;
import elemental.json.JsonValue;

/**
 * A renderer for presenting Components.
 * <p>
 * <strong>Note:</strong> The use of ComponentRenderer causes the Grid to
 * generate components for all items currently available in the client-side.
 * This means that a number of components is always generated and sent to the
 * client. Using complex structures of many nested components might be heavy to
 * generate and store, which will lead to performance problems.
 * <p>
 * <strong>Note:</strong> Components will occasionally be generated again during
 * runtime e.g. when selection changes. If your component has an internal state
 * that is not stored into the object, you should reuse the same component
 * instances.
 * <p>
 * Example of how to add a {@link Label} component to {@link Grid}:
 *
 * <pre>
 * Grid&lt;Person&gt; grid;
 * grid.addColumn(person -&gt; new Label(person.getFullName()),
 *         new ComponentRenderer()).setCaption("Full Name");
 * </pre>
 *
 * @author Vaadin Ltd
 * @since 8.1
 */
public class ComponentRenderer extends AbstractRenderer<Object, Component> {

    /**
     * Constructor for ComponentRenderer.
     */
    public ComponentRenderer() {
        super(Component.class);
    }

    @Override
    public JsonValue encode(Component value) {
        if (value instanceof AbstractField && getParentGrid().isReadOnly()) {
            ((AbstractField<?>) value).setReadOnly(true);
        }
        return value != null ? Json.create(value.getConnectorId()) : null;
    }

    @Override
    protected ComponentRendererState getState(boolean markAsDirty) {
        return (ComponentRendererState) super.getState(markAsDirty);
    }

    @Override
    protected ComponentRendererState getState() {
        return (ComponentRendererState) super.getState();
    }
}
