/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.data;

import java.io.Serializable;
import java.util.Optional;

import com.vaadin.server.Setter;

/**
 * A property from a {@link PropertySet}.
 *
 * @author Vaadin Ltd
 * @since 8.0
 *
 * @param <T>
 *            the type of the property set
 * @param <V>
 *            the property type
 */
public interface PropertyDefinition<T, V> extends Serializable {
    /**
     * Gets the value provider that is used for finding the value of this
     * property for a bean.
     *
     * @return the getter, not <code>null</code>
     */
    public ValueProvider<T, V> getGetter();

    /**
     * Gets an optional setter for storing a property value in a bean.
     *
     * @return the setter, or an empty optional if this property is read-only
     */
    public Optional<Setter<T, V>> getSetter();

    /**
     * Gets the type of this property.
     *
     * @return the property type. not <code>null</code>
     */
    public Class<V> getType();

    /**
     * Gets the type of the class containing this property.
     *
     * @since 8.1
     *
     * @return the property type. not <code>null</code>
     */
    public Class<?> getPropertyHolderType();

    /**
     * Gets the full name of this property.
     *
     * @return the property name, not <code>null</code>
     */
    public String getName();

    /**
     * Gets the top level name of this property.
     *
     * @return the top level property name, not <code>null</code>
     * @since 8.3
     */
    public default String getTopLevelName() {
        return getName();
    }

    /**
     * Gets the human readable caption to show for this property.
     *
     * @return the caption to show, not <code>null</code>
     */
    public String getCaption();

    /**
     * Gets the {@link PropertySet} that this property belongs to.
     *
     * @return the property set, not <code>null</code>
     */
    public PropertySet<T> getPropertySet();
}
