/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.navigator;

import java.util.EventObject;

/**
 * Event sent to the View instance before navigating away from it.
 * <p>
 * Provides a {@link #navigate()} method which must be called for the navigation
 * to take place.
 *
 * @since 8.1
 */
public class ViewBeforeLeaveEvent extends EventObject {

    private ViewLeaveAction action;
    private boolean navigateRun = false;

    /**
     * Creates a new event instance for the given navigator.
     *
     * @param navigator
     *            the navigator instance
     * @param action
     *            the command to execute when calling {@link #navigate()}
     */
    public ViewBeforeLeaveEvent(Navigator navigator, ViewLeaveAction action) {
        super(navigator);
        this.action = action;
    }

    /**
     * Performs the navigation which triggered the event in the first place.
     */
    public void navigate() {
        if (navigateRun) {
            throw new IllegalStateException(
                    "navigate() can only be called once");
        }
        action.run();
        navigateRun = true;
    }

    /**
     * Checks if the navigate command has been executed.
     *
     * @return <code>true</code> if {@link #navigate()} has been called,
     *         <code>false</code> otherwise
     */
    protected boolean isNavigateRun() {
        return navigateRun;
    }
}
