/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.data;

import java.lang.annotation.Annotation;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import com.vaadin.server.SerializablePredicate;

/**
 * This interface represents a predicate which returns {@code true} if bound
 * field should be configured to have required indicator via
 * {@link HasValue#setRequiredIndicatorVisible(boolean)}.
 *
 * @see BeanValidationBinder
 * @see BeanValidationBinder#setRequiredConfigurator(RequiredFieldConfigurator)
 *
 * @author Vaadin Ltd
 * @since 8.0
 *
 */
public interface RequiredFieldConfigurator
        extends SerializablePredicate<Annotation> {

    /**
     * Configurator which is aware of {@literal @NotNull} annotation presence
     * for a property.
     */
    public RequiredFieldConfigurator NOT_NULL = annotation -> annotation
            .annotationType().equals(NotNull.class);

    /**
     * Configurator which is aware of {@literal @NotEmpty} annotation presence
     * for a property.
     */
    public RequiredFieldConfigurator NOT_EMPTY = annotation -> annotation
            .annotationType().getName()
            .equals("org.hibernate.validator.constraints.NotEmpty") || annotation
            .annotationType().getName().equals("javax.validation.constraints.NotEmpty");

    /**
     * Configurator which is aware of {@literal Size} annotation with
     * {@code min()> 0} presence for a property.
     */
    public RequiredFieldConfigurator SIZE = annotation -> annotation
            .annotationType().equals(Size.class)
            && ((Size) annotation).min() > 0;

    /**
     * Default configurator which is combination of {@link #NOT_NULL},
     * {@link #NOT_EMPTY} and {@link #SIZE} configurators.
     */
    public RequiredFieldConfigurator DEFAULT = NOT_NULL.chain(NOT_EMPTY)
            .chain(SIZE);

    public default RequiredFieldConfigurator chain(
            RequiredFieldConfigurator configurator) {
        return descriptor -> test(descriptor) || configurator.test(descriptor);
    }
}
