/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.components.grid;

import java.util.Collections;
import java.util.List;

import com.vaadin.shared.ui.dnd.EffectAllowed;
import com.vaadin.ui.Grid;
import com.vaadin.ui.dnd.event.DragStartEvent;

/**
 * Drag start event on an HTML5 drag source {@link Grid} row.
 *
 * @param <T>
 *            The Grid bean type.
 * @author Vaadin Ltd.
 * @see GridDragSource#addGridDragStartListener(GridDragStartListener)
 * @since 8.1
 */
public class GridDragStartEvent<T> extends DragStartEvent<Grid<T>> {

    private final List<T> draggedItems;

    /**
     * Creates a drag start event.
     *
     * @param source
     *            The source grid where the rows are being dragged from.
     * @param effectAllowed
     *            Allowed effect from {@code DataTransfer.effectAllowed} object.
     * @param draggedItems
     *            List of items being dragged.
     */
    public GridDragStartEvent(Grid<T> source, EffectAllowed effectAllowed,
            List<T> draggedItems) {
        super(source, effectAllowed);

        this.draggedItems = draggedItems;
    }

    /**
     * Get the dragged row items.
     * <p>
     * The ordering of the list is the following: first the item that the drag
     * started from, optionally followed by all the other selected rows in
     * first-to-last order on the client side.
     *
     * @return an unmodifiable list of items that are being dragged.
     */
    public List<T> getDraggedItems() {
        return Collections.unmodifiableList(draggedItems);
    }
}
