/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.testbench.elements;

import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebElement;

import com.vaadin.testbench.By;
import com.vaadin.testbench.elementsbase.ServerClass;

@ServerClass("com.vaadin.ui.RichTextArea")
public class RichTextAreaElement extends AbstractFieldElement {

    /**
     * Gets the {@code <iframe>} element inside the component, containing the
     * editor.
     *
     * @return the iframe element containing the editor
     * @since 8.1.1
     */
    public WebElement getEditorIframe() {
        return findElement(By.tagName("iframe"));
    }

    /**
     * Return value of the field element.
     *
     * @return value of the field element
     * @since 8.4
     */
    public String getValue() {
        JavascriptExecutor executor = (JavascriptExecutor) getDriver();
        return executor.executeScript(
                "return arguments[0].contentDocument.body.innerHTML",
                getEditorIframe()).toString();
    }

    /**
     * Set value of the field element.
     *
     * @param chars
     *            new value of the field
     * @since 8.4
     */
    public void setValue(CharSequence chars) throws ReadOnlyException {
        if (isReadOnly()) {
            throw new ReadOnlyException();
        }
        JavascriptExecutor executor = (JavascriptExecutor) getDriver();
        executor.executeScript("var bodyE=arguments[0].contentDocument.body;\n"
                + "bodyE.innerHTML=arguments[1]; \n"
                + "var ev = document.createEvent('HTMLEvents');\n"
                + "ev.initEvent('change', true, false); \n"
                + "bodyE.dispatchEvent(ev);", getEditorIframe(), chars);
    }

    @Override
    public void focus() {
        waitForVaadin();
        JavascriptExecutor executor = (JavascriptExecutor) getDriver();
        executor.executeScript("arguments[0].contentDocument.body.focus();",
                getEditorIframe());
    }
}
