package com.vendasta.common.v1;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import com.vendasta.common.v1.CredentialsManager;
import com.vendasta.common.v1.SDKException;

import io.grpc.*;
import io.grpc.stub.MetadataUtils;

public abstract class SDKClient<T extends io.grpc.stub.AbstractStub<T>> {

    static final Metadata.Key<String> AUTHORIZATION = Metadata.Key.of("Authorization", Metadata.ASCII_STRING_MARSHALLER);
    
    private final int port = 443;
    private final String host;
    private final ManagedChannel channel;
    
    protected T blockingStub;
    
    public SDKClient(String host, InputStream serviceAccount) throws com.vendasta.common.v1.SDKException {
        this.host = host;
    
        try {
            this.channel = ManagedChannelBuilder.forAddress(this.host, this.port).build();
            CredentialsManager credentialsManager = this.getCredentialsManager(serviceAccount, host);
            T stub = this.newBlockingStub(channel).withCallCredentials(new OAuthCallCredentials(credentialsManager));
            this.blockingStub = stub.withWaitForReady();
        } catch(Exception e) {
            throw new SDKException(e);
        }
    }

    public void shutdown() throws SDKException {
        try {
        channel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            throw new SDKException("Error shutting down channel", e);
        }
    }
    /** 
     * @return The credentials manager to use to get the authorization header
     * @throws IOException if there's an issue with the input stream
     */
    protected abstract CredentialsManager getCredentialsManager(InputStream serviceAccount, String host) throws IOException;
    
    /**
     * @param channel the channel to use for the blocking stub
     * This should create a new blocking stub and set the instance property to be used later
     * It should also set the headers
     */
    protected abstract T newBlockingStub(ManagedChannel channel);

    private class OAuthCallCredentials implements CallCredentials {
        private CredentialsManager credentialsManager;

        OAuthCallCredentials(CredentialsManager credentialsManager) {
            this.credentialsManager = credentialsManager;
        }

        @Override
        public void applyRequestMetadata(MethodDescriptor<?, ?> methodDescriptor, Attributes attributes, Executor executor, final MetadataApplier metadataApplier) {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        Metadata headers = new Metadata();
                        headers.put(SDKClient.AUTHORIZATION, credentialsManager.getAuthorization());
                        metadataApplier.apply(headers);
                    } catch (Throwable e) {
                        metadataApplier.fail(Status.UNAUTHENTICATED.withCause(e));
                    }
                }
            });
        }
    }
}