package com.vendasta.sso.v1;

import org.apache.commons.codec.binary.Base64;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.annotations.SerializedName;
import com.vendasta.sso.v1.internal.GetEntryURLRequest;
import com.vendasta.sso.v1.internal.GetEntryURLResponse;
import com.vendasta.sso.v1.internal.GetEntryURLWithCodeRequest;
import com.vendasta.sso.v1.internal.GetEntryURLWithCodeResponse;
import com.vendasta.sso.v1.internal.IdentityProviderGeneratedClient;
import com.vendasta.sso.v1.internal.LogoutRequest;
import com.vendasta.vax.Environment;
import com.vendasta.vax.SDKException;

import java.io.InputStream;


/**
 * Provides functions to implement the Identity Provider for <a href="https://www.vendasta.com/developers/partners/sso">Vendasta sso</a>
 * Authentication is done by reading the path set in the environment variable <pre>VENDASTA_APPLICATION_CREDENTIALS</pre> unless the service account
 * has been specified as an input stream to the constructor
 * This env var should be a path pointing to your Service Account Private Key that you have previously downloaded (as described <a href="https://www.vendasta.com/developers/service-accounts">here</a>)
 */
public class IdentityProviderClient extends IdentityProviderGeneratedClient {
	private static Gson gson = new Gson();

	/**
	 * Constructs an identity provider client that will hit the environment provided.
     * * Will use the VENDASTA_APPLICATION_CREDENTIALS environment param
	 * @param environment The environment for the api calls coming out of this class to go to.
	 */
	public IdentityProviderClient(Environment environment) {
        super(environment);
    }
	
	/**
	 * Constructs an identity provider client that will hit the environment provided.
     * Will use the VENDASTA_APPLICATION_CREDENTIALS environment param
	 * @param environment The environment for the api calls coming out of this class to go to.
	 * @param defaultTimeout The duration in milliseconds that the sdk will cancel the api call after.
	 */
    public IdentityProviderClient(Environment environment, float defaultTimeout) {
        super(environment, defaultTimeout);
    }

    /**
     * Constructs an identity provider client that will hit the environment provided.
     * @param environment The environment for the api calls coming out of this class to go to.
     * @param serviceAccount The input stream of the service account to authenticate the requests with
     * @param defaultTimeout The duration in milliseconds that the sdk will cancel the api call after.
     */
    public IdentityProviderClient(Environment environment, InputStream serviceAccount, float defaultTimeout) {
        super(environment, serviceAccount, defaultTimeout);
    }

    /**
     * Constructs an identity provider client that will hit the environment provided.
     * @param environment The environment for the api calls coming out of this class to go to.
     * @param serviceAccount The input stream of the service account to authenticate the requests with
     * @param options The Options for the client, ie whether or not to use gRPC, or setting a default timeout
     */
    public IdentityProviderClient(Environment environment, InputStream serviceAccount, Options options) {
        super(environment, serviceAccount, options);
    }

    /**
     * A context indicating SSO is being requested for a particular account
     */
    public static final class AccountContext {

        @SerializedName("account_id")
        private final String accountId;

        public AccountContext(final String accountId) {
            this.accountId = accountId;
        }

        public String getAccountId() {
            return this.accountId;
        }
    }

    /**
     * A context indicating SSO is being requested for a particular brand
     */
    public static final class BrandContext {

        @SerializedName("group_path")
        private final String groupPath;

        public BrandContext(final String groupPath) {
            this.groupPath = groupPath;
        }

        public String getGroupPath() {
            return this.groupPath;
        }

    }

    /**
     * A context indicating SSO is being requested for a particular partner (not tied to a single account, but still
     * scoped to a single partner)
     */
    public static final class PartnerContext {

        @SerializedName("partner_id")
        private final String partnerId;

        public PartnerContext(final String partnerId) {
            this.partnerId = partnerId;
        }

        public String getPartnerId() {
            return this.partnerId;
        }

    }

    /**
     * Encapsulates the different types of contexts that can be decoded from the base64 encoded 'serviceContext' query
     * param
     */
    public final class ServiceContext {
        private AccountContext accountContext = null;
        private PartnerContext partnerContext = null;
        private BrandContext brandContext = null;

        public AccountContext getAccountContext() {
            return this.accountContext;
        }

        public PartnerContext getPartnerContext() {
            return this.partnerContext;
        }

        public BrandContext getBrandContext() {
            return this.brandContext;
        }


        public ServiceContext(final String b64EncodedContext) {
            String decoded = new String(Base64.decodeBase64(b64EncodedContext.getBytes()));
            try {
                String type = determineContextType(decoded);
                if (type.equals("account")) {
                    this.accountContext = gson.fromJson(decoded, AccountContext.class);
                } else if (type.equals("partner")) {
                    this.partnerContext = gson.fromJson(decoded, PartnerContext.class);
                } else if (type.equals("brand")) {
                    this.brandContext = gson.fromJson(decoded, BrandContext.class);
                } else {
                    throw new UnknownContextException("unknown context: " + type);
                }
            } catch (JsonSyntaxException e) {
                throw new JsonParseException("unable to parse json for context: " + b64EncodedContext);
            }
        }

        private class ContextType {
            private String _type;
        }

        private String determineContextType(String json) throws InvalidContextException {
            Gson gson = new Gson();
            ContextType ctype = gson.fromJson(json, ContextType.class);
            if (ctype == null) {
                throw new InvalidContextException();
            }
            return ctype._type;
        }
    }
    
    public class UnknownContextException extends SDKException {
        public UnknownContextException(String message) {
            super(message);
        }
    }

    public class JsonParseException extends SDKException {
        public JsonParseException(String message) {
            super(message);
        }
    }

    public class InvalidContextException extends SDKException {
        public InvalidContextException() {
            super("Invalid context. The context may not be valid json, or it may be missing the '_type' field");
        }
    }

    public class SessionNotFoundException extends SDKException {
        public SessionNotFoundException() {
            super("Session not found for the given session ID.");
        }
    }

    /**
     * RPC for getting the entry URL of a service provider
     *
     * @param serviceProviderID The ID of the service provider (such as a marketplace application)
     * @param encodedContext    A base64 encoded JSON string representing a particular ServiceContext, provided in the
     *                          serviceContext query param of a session transfer handler request. The context helps
     *                          form the entry URL by providing parametrized fields, such as an account ID.
     * @return The entry URL
     * @throws SDKException            Any SDK exception
     * @throws UnknownContextException when the context type cannot be determined, or the context is corrupt
     * @throws JsonParseException      when the context's content is not parsable json
     */
    public String getEntryURL(String serviceProviderID, String encodedContext) throws SDKException {
        ServiceContext decodedContext = new ServiceContext(encodedContext);
        if (decodedContext.getAccountContext() != null) {
            return getEntryURL(serviceProviderID, decodedContext.getAccountContext());
        } else if (decodedContext.getPartnerContext() != null) {
            return getEntryURL(serviceProviderID, decodedContext.getPartnerContext());
        }
        throw new UnknownContextException("unknown context");
    }

    /**
     * RPC for getting the entry URL of a service provider
     *
     * @param serviceProviderID The ID of the service provider (such as a marketplace application)
     * @param context           The context helps form the entry URL by providing parametrized fields, such as an
     *                          account ID
     * @return The entry URL
     * @throws SDKException Any SDK exception
     */
    public String getEntryURL(String serviceProviderID, AccountContext context) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext =
        		com.vendasta.sso.v1.internal.ServiceContext.newBuilder()
                        .setAccount(com.vendasta.sso.v1.internal.ServiceContext.Account.newBuilder()
                                .setAccountId(context.getAccountId())
                                .build())
                        .build();

        return this.getEntryURL(serviceProviderID, serviceContext);
    }

    /**
     * RPC for getting the entry URL of a service provider
     *
     * @param serviceProviderID The ID of the service provider (such as a marketplace application)
     * @param context           The context helps form the entry URL by providing parametrized fields, such as an
     *                          account ID
     * @return The entry URL
     * @throws SDKException Any SDK exception
     */
    public String getEntryURL(String serviceProviderID, BrandContext context) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext =
        		com.vendasta.sso.v1.internal.ServiceContext.newBuilder()
                        .setBrand(com.vendasta.sso.v1.internal.ServiceContext.Brand.newBuilder()
                                .setGroupPath(context.getGroupPath())
                                .build())
                        .build();

        return this.getEntryURL(serviceProviderID, serviceContext);
    }

    /**
     * RPC for getting the entry URL of a service provider
     *
     * @param serviceProviderID The ID of the service provider (such as a marketplace application)
     * @param context           The context helps form the entry URL by providing parametrized fields, such as a
     *                          partner ID
     * @return The entry URL
     * @throws SDKException Any SDK exception
     */
    public String getEntryURL(String serviceProviderID, PartnerContext context) throws SDKException {
    	com.vendasta.sso.v1.internal.ServiceContext serviceContext =
    			com.vendasta.sso.v1.internal.ServiceContext.newBuilder()
                        .setPartner(com.vendasta.sso.v1.internal.ServiceContext.Partner.newBuilder()
                                .setPartnerId(context.getPartnerId())
                                .build())
                        .build();

        return this.getEntryURL(serviceProviderID, serviceContext);
    }

    private String getEntryURL(String serviceProviderID, com.vendasta.sso.v1.internal.ServiceContext context) throws SDKException {
        GetEntryURLRequest request = GetEntryURLRequest.newBuilder()
                .setServiceProviderId(serviceProviderID)
                .setContext(context)
                .build();
        GetEntryURLResponse resp = this.GetEntryURL(request, null);
        return resp.getEntryUrl();
    }


    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param encodedContext    A base64 encoded JSON string representing a particular ServiceContext, provided in the
     *                          serviceContext query param of a session transfer handler request. The context helps
     *                          form the entry URL by providing parametrized fields, such as an account ID.
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @return The entry URL, with the authentication code attached
     * @throws SDKException            Any SDK exception
     * @throws UnknownContextException when the context type cannot be determined, or the context is corrupt
     * @throws JsonParseException      when the context's content is not parsable json
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      String encodedContext, String nextURL) throws SDKException {
        ServiceContext decodedContext = new ServiceContext(encodedContext);
        if (decodedContext.getAccountContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getAccountContext(), nextURL);
        } else if (decodedContext.getPartnerContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getPartnerContext(), nextURL);
        } else if (decodedContext.getBrandContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getBrandContext(), nextURL);
        }
        throw new UnknownContextException("unknown context");
    }

    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be pass
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param encodedContext    A base64 encoded JSON string representing a particular ServiceContext, provided in the
     *                          serviceContext query param of a session transfer handler request. The context helps
     *                          form the entry URL by providing parametrized fields, such as an account ID.
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @param backUrl           The Back URL to be used in the navigation bar
     * @param backUrlText       The next URL Text to be used in the navigation bar
     * @return The entry URL, with the authentication code attached
     * @throws SDKException            Any SDK exception
     * @throws UnknownContextException when the context type cannot be determined, or the context is corrupt
     * @throws JsonParseException      when the context's content is not parsable json
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      String encodedContext, String nextURL, String backUrl, String backUrlText) throws SDKException {
        ServiceContext decodedContext = new ServiceContext(encodedContext);
        if (decodedContext.getAccountContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getAccountContext(), nextURL, backUrl, backUrlText);
        } else if (decodedContext.getPartnerContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getPartnerContext(), nextURL, backUrl, backUrlText);
        } else if (decodedContext.getBrandContext() != null) {
            return getEntryURLWithCode(serviceProviderID, sessionID, userID, email, decodedContext.getBrandContext(), nextURL, backUrl, backUrlText);
        }
        throw new UnknownContextException("unknown context");
    }

    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as an
     *                          account ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      AccountContext context, String nextURL) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder()
                .setAccount(com.vendasta.sso.v1.internal.ServiceContext.Account.newBuilder()
                        .setAccountId(context.getAccountId())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, "", "");
    }

    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be pass
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as an
     *                          account ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @param backUrl           The Back URL to be used in the navigation bar
     * @param backUrlText       The next URL Text to be used in the navigation bar
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      AccountContext context, String nextURL, String backUrl, String backUrlText) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder()
                .setAccount(com.vendasta.sso.v1.internal.ServiceContext.Account.newBuilder()
                        .setAccountId(context.getAccountId())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, backUrl, backUrlText);
    }

    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be pass
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as a
     *                          partner ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      PartnerContext context, String nextURL) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder().
                setPartner(com.vendasta.sso.v1.internal.ServiceContext.Partner.newBuilder()
                        .setPartnerId(context.getPartnerId())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, "", "");

    }

    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be pass
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as a
     *                          partner ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @param backUrl           The Back URL to be used in the navigation bar
     * @param backUrlText       The next URL Text to be used in the navigation bar
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      PartnerContext context, String nextURL, String backUrl, String backUrlText) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder().
                setPartner(com.vendasta.sso.v1.internal.ServiceContext.Partner.newBuilder()
                        .setPartnerId(context.getPartnerId())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, backUrl, backUrlText);

    }


    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be passed
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as a
     *                          partner ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      BrandContext context, String nextURL) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder().
                setBrand(com.vendasta.sso.v1.internal.ServiceContext.Brand.newBuilder()
                        .setGroupPath(context.getGroupPath())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, "", "");

    }


    /**
     * RPC for getting the entry URL of the service provider, with the authentication code attached
     * This method is overloaded for allowing back url and back url text parameters to be passed
     *
     * @param serviceProviderID The ID of a service provider (such as a marketplace application)
     * @param sessionID         The user's session ID. This can be the session ID directly from your application, or it
     *                          can be a hashed version, or something else unique. Whatever it is, the same session ID
     *                          must be passed to the logout RPC.
     * @param userID            The user's ID
     * @param email             The user's email
     * @param context           The context helps form the entry URL by providing parametrized fields, such as a
     *                          partner ID
     * @param nextURL           The next URL to send the user to, once the code exchange is complete on the entry URL
     * @return The entry URL, with the authentication code attached
     * @throws SDKException Any SDK exception
     */
    public String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                      BrandContext context, String nextURL, String backUrl, String backUrlText) throws SDKException {
        com.vendasta.sso.v1.internal.ServiceContext serviceContext = com.vendasta.sso.v1.internal.ServiceContext.newBuilder().
                setBrand(com.vendasta.sso.v1.internal.ServiceContext.Brand.newBuilder()
                        .setGroupPath(context.getGroupPath())
                        .build())
                .build();

        return this.getEntryURLWithCode(serviceProviderID, sessionID, userID, email, serviceContext, nextURL, backUrl, backUrlText);

    }


    private String getEntryURLWithCode(String serviceProviderID, String sessionID, String userID, String email,
                                       com.vendasta.sso.v1.internal.ServiceContext serviceContext, String nextURL,
                                       String backUrl, String backUrlText)
            throws SDKException {
        GetEntryURLWithCodeRequest request = GetEntryURLWithCodeRequest.newBuilder()
                .setServiceProviderId(serviceProviderID)
                .setSessionId(sessionID)
                .setUserId(userID)
                .setEmail(email)
                .setContext(serviceContext)
                .setNextUrl(nextURL)
                .setBackUrl(backUrl)
                .setBackUrlText(backUrlText)
                .build();
    	GetEntryURLWithCodeResponse response = this.GetEntryURLWithCode(request, null);
        return response.getEntryUrl();
    }

    /**
     * RPC for logging the user out of all active sessions, regardless of service provider
     *
     * @param sessionID The session ID that was provided to the getEntryURLWithCode RPC
     * @throws SDKException Any SDK exception
     */
    public void logout(String sessionID) throws SDKException {
    	
        LogoutRequest request = LogoutRequest.newBuilder()
                .setSessionId(sessionID)
                .build();
        
        this.Logout(request, null);
    }
}
