/* Copyright (c) 2020 vesoft inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License.
 */

package org.apache.flink.connector.nebula.statement;

import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_BATCH_INTERVAL_MS;
import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_EXECUTION_RETRY;
import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_RETRY_DELAY_MS;
import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_ROW_INFO_INDEX;
import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_SCAN_LIMIT;
import static org.apache.flink.connector.nebula.utils.NebulaConstant.DEFAULT_WRITE_BATCH_SIZE;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.flink.connector.nebula.utils.DataTypeEnum;
import org.apache.flink.connector.nebula.utils.FailureHandlerEnum;
import org.apache.flink.connector.nebula.utils.PolicyEnum;
import org.apache.flink.connector.nebula.utils.WriteModeEnum;

public class EdgeExecutionOptions extends ExecutionOptions {

    /**
     * nebula edge type
     */
    private String edge;

    /**
     * src index for nebula edge sink
     */
    private int srcIndex;

    /**
     * dst index for nebula edge sink
     */
    private int dstIndex;

    /**
     * rank index for nebula edge sink
     */
    private int rankIndex;


    private EdgeExecutionOptions(String graphSpace,
                                 String executeStatement,
                                 List<String> fields,
                                 List<Integer> positions,
                                 boolean noColumn,
                                 int limit,
                                 long startTime,
                                 long endTime,
                                 int batch,
                                 PolicyEnum policy,
                                 WriteModeEnum mode,
                                 String edge,
                                 int srcIndex,
                                 int dstIndex,
                                 int rankIndex,
                                 int batchIntervalMs,
                                 FailureHandlerEnum failureHandler,
                                 int maxRetries,
                                 int retryDelayMs) {
        super(graphSpace, executeStatement, fields, positions, noColumn, limit, startTime,
                endTime, batch, policy, mode, batchIntervalMs,
                failureHandler, maxRetries, retryDelayMs);
        this.edge = edge;
        this.srcIndex = srcIndex;
        this.dstIndex = dstIndex;
        this.rankIndex = rankIndex;
    }

    public String getEdge() {
        return edge;
    }

    public int getSrcIndex() {
        return srcIndex;
    }

    public int getDstIndex() {
        return dstIndex;
    }

    public int getRankIndex() {
        return rankIndex;
    }

    @Override
    public String getLabel() {
        return edge;
    }

    @Override
    public DataTypeEnum getDataType() {
        return DataTypeEnum.EDGE;
    }

    public ExecutionOptionBuilder toBuilder() {
        return new ExecutionOptionBuilder()
                .setGraphSpace(this.getGraphSpace())
                .setExecuteStatement(this.getExecuteStatement())
                .setEdge(this.getEdge())
                .setFields(this.getFields())
                .setPositions(this.getPositions())
                .setNoColumn(this.isNoColumn())
                .setLimit(this.getLimit())
                .setStartTime(this.getStartTime())
                .setEndTime(this.getEndTime())
                .setBatchSize(this.getBatchSize())
                .setPolicy(Optional.ofNullable(this.getPolicy())
                        .map(Objects::toString).orElse(null))
                .setSrcIndex(this.getSrcIndex())
                .setDstIndex(this.getDstIndex())
                .setRankIndex(this.getRankIndex())
                .setWriteMode(this.getWriteMode())
                .setBatchIntervalMs(this.getBatchIntervalMs())
                .setFailureHandler(this.getFailureHandler())
                .setMaxRetries(this.getMaxRetries())
                .setRetryDelayMs(this.getRetryDelayMs());
    }

    public static class ExecutionOptionBuilder {
        private String graphSpace;
        private String executeStatement;
        private String edge;
        private List<String> fields;
        private List<Integer> positions;
        private boolean noColumn = false;
        private int limit = DEFAULT_SCAN_LIMIT;
        private long startTime = 0;
        private long endTime = Long.MAX_VALUE;
        private int batchSize = DEFAULT_WRITE_BATCH_SIZE;
        private int batchIntervalMs = DEFAULT_BATCH_INTERVAL_MS;
        private PolicyEnum policy = null;
        private WriteModeEnum mode = WriteModeEnum.INSERT;
        private int srcIndex = DEFAULT_ROW_INFO_INDEX;
        private int dstIndex = DEFAULT_ROW_INFO_INDEX;
        private int rankIndex = DEFAULT_ROW_INFO_INDEX;
        private FailureHandlerEnum failureHandler = FailureHandlerEnum.IGNORE;
        private int maxRetries = DEFAULT_EXECUTION_RETRY;
        private int retryDelayMs = DEFAULT_RETRY_DELAY_MS;

        public ExecutionOptionBuilder setGraphSpace(String graphSpace) {
            this.graphSpace = graphSpace;
            return this;
        }

        public ExecutionOptionBuilder setExecuteStatement(String executeStatement) {
            this.executeStatement = executeStatement;
            return this;
        }

        public ExecutionOptionBuilder setEdge(String edge) {
            this.edge = edge;
            return this;
        }

        public ExecutionOptionBuilder setFields(List<String> fields) {
            this.fields = fields;
            return this;
        }

        public ExecutionOptionBuilder setPositions(List<Integer> positions) {
            this.positions = positions;
            return this;
        }

        public ExecutionOptionBuilder setNoColumn(boolean noColumn) {
            this.noColumn = noColumn;
            return this;
        }

        public ExecutionOptionBuilder setLimit(int limit) {
            this.limit = limit;
            return this;
        }

        public ExecutionOptionBuilder setStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public ExecutionOptionBuilder setEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }

        public ExecutionOptionBuilder setBatchSize(int batchSize) {
            this.batchSize = batchSize;
            return this;
        }

        @Deprecated
        public ExecutionOptionBuilder setBatch(int batch) {
            return setBatchSize(batch);
        }

        public ExecutionOptionBuilder setPolicy(String policy) {
            if (policy != null && !policy.trim().isEmpty()) {
                this.policy = PolicyEnum.valueOf(policy);
            }
            return this;
        }

        public ExecutionOptionBuilder setSrcIndex(int srcIndex) {
            this.srcIndex = srcIndex;
            return this;
        }

        public ExecutionOptionBuilder setDstIndex(int dstIndex) {
            this.dstIndex = dstIndex;
            return this;
        }

        public ExecutionOptionBuilder setRankIndex(int rankIndex) {
            this.rankIndex = rankIndex;
            return this;
        }

        public ExecutionOptionBuilder setWriteMode(WriteModeEnum mode) {
            this.mode = mode;
            return this;
        }

        public ExecutionOptionBuilder setBatchIntervalMs(int batchIntervalMs) {
            this.batchIntervalMs = batchIntervalMs;
            return this;
        }

        public ExecutionOptionBuilder setFailureHandler(FailureHandlerEnum failureHandler) {
            this.failureHandler = failureHandler;
            return this;
        }

        public ExecutionOptionBuilder setMaxRetries(int maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public ExecutionOptionBuilder setRetryDelayMs(int retryDelayMs) {
            this.retryDelayMs = retryDelayMs;
            return this;
        }

        @Deprecated
        public EdgeExecutionOptions builder() {
            return build();
        }

        public EdgeExecutionOptions build() {
            if (graphSpace == null || graphSpace.trim().isEmpty()) {
                throw new IllegalArgumentException("graph space can not be empty.");
            }
            if (edge == null || edge.trim().isEmpty()) {
                throw new IllegalArgumentException("edge can not be empty.");
            }
            return new EdgeExecutionOptions(graphSpace, executeStatement, fields, positions,
                    noColumn, limit, startTime, endTime, batchSize, policy, mode, edge, srcIndex,
                    dstIndex, rankIndex, batchIntervalMs, failureHandler, maxRetries, retryDelayMs);
        }
    }
}
