package com.virjar.vtoolkit;


import com.alibaba.fastjson.JSONObject;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * 对系统内部的运行状态进行审查，为实时审查接口
 */
public class Inspector {


    private static final Map<String, Collector> allCollectors = new ConcurrentHashMap<>();

    public static void addCollector(String topic, Collector collector) {
        allCollectors.put(topic, collector);
    }

    public static void removeCollector(String topic) {
        allCollectors.remove(topic);
    }


    public interface Collector {
        void collect(Callback callback);
    }

    public static class Callback {
        private final String key;
        private final CountDownLatch countDownLatch;
        private Object value;

        Callback(String key, CountDownLatch countDownLatch) {
            this.key = key;
            this.countDownLatch = countDownLatch;
        }

        public void onData(Object data) {
            this.value = data;
            countDownLatch.countDown();
        }
    }

    public abstract static class Joiner {
        private final Map<String, Collector> map = new ConcurrentHashMap<>();
        private final JSONObject joinerData = new JSONObject();

        public Joiner add(String topic, Collector collector) {
            map.put(topic, collector);
            return this;
        }

        public void doJoin() {
            doJoin(map.entrySet().iterator());
        }

        private void doJoin(Iterator<Map.Entry<String, Collector>> iterator) {
            if (!iterator.hasNext()) {
                onFinish(joinerData);
                return;
            }
            Map.Entry<String, Collector> entry = iterator.next();
            String topic = entry.getKey();
            entry.getValue().collect(new Callback(topic, null) {
                @Override
                public void onData(Object data) {
                    joinerData.put(topic, data);
                    doJoin(iterator);
                }
            });
        }

        public abstract void onFinish(JSONObject jsonObject);
    }

    public static JSONObject inspect() {
        CountDownLatch countDownLatch = new CountDownLatch(allCollectors.size());
        List<Callback> callbacks = new ArrayList<>(allCollectors.size());

        for (Map.Entry<String, Collector> entry : allCollectors.entrySet()) {
            Callback callback = new Callback(entry.getKey(), countDownLatch);
            callbacks.add(callback);
            entry.getValue().collect(callback);
        }

        try {
            countDownLatch.await(30, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        JSONObject jsonObject = new JSONObject(new TreeMap<>());
        for (Callback callback : callbacks) {
            Object value = callback.value;
            if (value == null) {
                value = "inspect failed!!";
            }
            jsonObject.put(callback.key, value);
        }
        return jsonObject;
    }

}

