package com.virjar.vtoolkit;


import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;

import javax.net.ssl.*;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 基于java原生的http访问
 */
@Slf4j
public class SimpleHttpInvoker {

    // 请求参数
    // 请求头部
    private static final ThreadLocal<LinkedHashMap<String, String>> threadLocalRequestHeader = new ThreadLocal<>();
    // 127.0.0.1:8888
    private static final ThreadLocal<String> threadLocalProxy = new ThreadLocal<>();
    // 账号代理的账号密码
    private static final ThreadLocal<AuthHolder> threadLocalProxyAuth = new ThreadLocal<>();

    private static final ThreadLocal<DefaultHttpPropertyBuilder> threadLocalDefaultHttpProperty = new ThreadLocal<>();

    private static final ThreadLocal<Integer> threadLocalConnectionTimeout = new ThreadLocal<>();

    private static final ThreadLocal<Integer> threadLocalReadTimeout = new ThreadLocal<>();

    // 响应参数
    private static final ThreadLocal<Integer> threadLocalResponseStatus = new ThreadLocal<>();
    private static final ThreadLocal<LinkedHashMap<String, String>> threadLocalResponseHeader = new ThreadLocal<>();
    private static final ThreadLocal<IOException> threadLocalResponseIOException = new ThreadLocal<>();

    public static String get(String url) {
        return asString(execute("GET", url, null));
    }

    public static byte[] getEntity(String url) {
        return execute("GET", url, null);
    }

    public static String post(String url, JSONObject body) {
        addHeader("Content-Type", "application/json; charset=UTF-8");
        return asString(execute("POST", url, body.toJSONString().getBytes(StandardCharsets.UTF_8)));
    }

    public static String post(String url, Map<String, String> body) {
        addHeader("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
        try {
            StringBuilder sb = new StringBuilder();
            for (Map.Entry<String, String> entry : body.entrySet()) {
                sb.append(URLEncoder.encode(entry.getKey(), StandardCharsets.UTF_8.name()))
                        .append("=");
                String value = entry.getValue();
                if (value != null) {
                    sb.append(URLEncoder.encode(value, StandardCharsets.UTF_8.name()))
                            .append("=");
                }
                sb.append("&");
            }
            if (sb.length() > 0) {
                sb.setLength(sb.length() - 1);
            }
            return asString(execute("POST", url, sb.toString().getBytes(StandardCharsets.UTF_8)));
        } catch (UnsupportedEncodingException unsupportedEncodingException) {
            // not happen
            throw new IllegalStateException(unsupportedEncodingException);
        }

    }

    public static void setProxy(String ip, String port) {
        threadLocalProxy.set(ip + ":" + port);
    }

    /**
     * 请在main函数中增加如下代码：
     * System.setProperty("jdk.http.auth.tunneling.disabledSchemes", "");
     * 解决jdk8不让https认证密码的问题，
     * 请注意一定要尽早添加，否则当http url connection组建完成初始化之后，该配置将会失效
     * {@link "https://gitee.com/dromara/hutool/issues/I1YQGM"}
     */
    public static void setProxyAuth(String userName, String password) {
        threadLocalProxyAuth.set(new AuthHolder(userName, password));
    }

    public static void setTimout(int connectTimout, int readTimeout) {
        threadLocalConnectionTimeout.set(connectTimout);
        threadLocalReadTimeout.set(readTimeout);
    }

    public static void addHeader(Map<String, String> headers) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            addHeader(entry.getKey(), entry.getValue());
        }
    }

    public static void addHeader(String key, String value) {
        LinkedHashMap<String, String> map = threadLocalRequestHeader.get();
        if (map == null) {
            map = new LinkedHashMap<>();
            threadLocalRequestHeader.set(map);
        }
        map.put(key, value);
    }

    public static IOException getIoException() {
        return threadLocalResponseIOException.get();
    }

    public static int getResponseStatus() {
        return threadLocalResponseStatus.get();
    }

    public static LinkedHashMap<String, String> getResponseHeader() {
        return threadLocalResponseHeader.get();
    }

    public static void setupDefaultHttpProperty(DefaultHttpPropertyBuilder defaultHttpPropertyBuilder) {
        threadLocalDefaultHttpProperty.set(defaultHttpPropertyBuilder);
        defaultHttpPropertyBuilder.build(newBuilder());
    }

    public static byte[] execute(String method, String url, byte[] body) {
        // reset response
        threadLocalResponseStatus.remove();
        threadLocalResponseHeader.remove();
        threadLocalResponseIOException.remove();

        // prepare proxy
        String proxyConfig = threadLocalProxy.get();
        HttpURLConnection connection;
        try {
            URL urlMode = new URL(url);
            if (proxyConfig != null && !proxyConfig.trim().isEmpty()) {
                // 有代理配置
                String[] ipAndPort = proxyConfig.trim().split(":");
                Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(
                        ipAndPort[0].trim(), Integer.parseInt(ipAndPort[1].trim())
                ));
                connection = (HttpURLConnection) urlMode.openConnection(proxy);
//                String authConfig = threadLocalProxyAuth.get();
//                if (authConfig != null && !authConfig.trim().isEmpty()) {
//                    addHeader("Proxy-Authorization", authConfig);
//                }
            } else {
                // 没有代理
                connection = (HttpURLConnection) urlMode.openConnection();
            }

            connection.setRequestMethod(method);

            Integer connectionTimeout = threadLocalConnectionTimeout.get();
            if (connectionTimeout == null) {
                connectionTimeout = 5000;
            }
            connection.setConnectTimeout(connectionTimeout);

            Integer readTimeout = threadLocalReadTimeout.get();
            if (readTimeout == null) {
                readTimeout = 5000;
            }
            // 一定要设置超时时间，在有代理的情况下，java的默认超时时间简直太长了
            connection.setReadTimeout(readTimeout);

            // fill http header
            LinkedHashMap<String, String> requestHeaders = threadLocalRequestHeader.get();
            if (requestHeaders != null) {
                for (Map.Entry<String, String> entry : requestHeaders.entrySet()) {
                    connection.setRequestProperty(entry.getKey(), entry.getValue());
                }
            }

            if (body != null) {
                connection.setDoOutput(true);
                try (OutputStream os = connection.getOutputStream()) {
                    os.write(body);
                }
            }
            return readResponse(connection);
        } catch (IOException e) {
            threadLocalResponseIOException.set(e);
            return null;
        } finally {
            // clear request
            threadLocalRequestHeader.remove();
            threadLocalProxy.remove();

            DefaultHttpPropertyBuilder builder = threadLocalDefaultHttpProperty.get();
            if (builder != null) {
                builder.build(newBuilder());
            }
        }
    }


    public static String asString(byte[] data) {
        if (data == null) {
            return null;
        }
        LinkedHashMap<String, String> headers = threadLocalResponseHeader.get();
        Charset charset = StandardCharsets.UTF_8;

        //content-type: application/json
        String contentType = headers.get("content-type");
        if (contentType != null && contentType.contains(":")) {
            String charsetStr = contentType.split(":")[1].trim();
            charset = Charset.forName(charsetStr);
        }
        return new String(data, charset);
    }

    private static byte[] readResponse(HttpURLConnection connection) throws IOException {
        threadLocalResponseStatus.set(connection.getResponseCode());
        LinkedHashMap<String, String> responseHeader = new LinkedHashMap<>();
        for (int i = 0; i < 128; i++) {
            String key = connection.getHeaderFieldKey(i);
            if (key == null) {
                if (i == 0) {
                    continue;
                } else {
                    break;
                }
            }
            responseHeader.put(key, connection.getHeaderField(key));
        }
        threadLocalResponseHeader.set(responseHeader);

        try (InputStream inputStream = connection.getInputStream()) {
            return IOUtils.toByteArray(inputStream);
        } finally {
            connection.disconnect();
        }
    }


    static {
        try {
            initSSL();
        } catch (Exception e) {
            throw new IllegalStateException("error", e);
        }
        setupAuthenticator();
    }

    private static void setupAuthenticator() {
        Authenticator.setDefault(new Authenticator() {
            @Override
            protected PasswordAuthentication getPasswordAuthentication() {
                //return super.getPasswordAuthentication();
                AuthHolder authHolder = threadLocalProxyAuth.get();
                if (authHolder == null) {
                    return super.getPasswordAuthentication();
                }
                return new PasswordAuthentication(authHolder.user, authHolder.pass);
            }
        });
    }

    private static class AuthHolder {
        private final String user;
        private final char[] pass;

        public AuthHolder(String user, String pass) {
            this.user = user;
            this.pass = pass.toCharArray();
        }
    }


    private static void initSSL() throws NoSuchAlgorithmException, KeyManagementException {
        TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager() {
            @Override
            public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                return null;
            }

            @Override
            public void checkClientTrusted(X509Certificate[] certs, String authType) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] certs, String authType) {
            }
        }
        };
        SSLContext sc = SSLContext.getInstance("SSL");
        sc.init(null, trustAllCerts, new java.security.SecureRandom());
        HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
        HostnameVerifier allHostsValid = (hostname, session) -> true;
        HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
    }

    public static RequestBuilder newBuilder() {
        return new RequestBuilder();
    }

    public static class RequestBuilder {

        public RequestBuilder setProxy(String ip, String port) {
            SimpleHttpInvoker.setProxy(ip, port);
            return this;
        }

        public RequestBuilder addHeader(Map<String, String> headers) {
            SimpleHttpInvoker.addHeader(headers);
            return this;
        }

        public RequestBuilder addHeader(String key, String value) {
            SimpleHttpInvoker.addHeader(key, value);
            return this;
        }

        public RequestBuilder setProxyAuth(String userName, String password) {
            SimpleHttpInvoker.setProxyAuth(userName, password);
            return this;
        }

        public RequestBuilder setTimout(int connectTimout, int readTimeout) {
            SimpleHttpInvoker.setTimout(connectTimout, readTimeout);
            return this;
        }

    }

    public interface DefaultHttpPropertyBuilder {
        void build(RequestBuilder requestBuilder);
    }


}

