package com.virjar.vtoolkit.trace;

import com.virjar.vtoolkit.StringSplitter;
import com.virjar.vtoolkit.ThrowablePrinter;
import com.virjar.vtoolkit.safethread.Looper;
import lombok.Getter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.lang.ref.WeakReference;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 他本来应该是一个枚举
 */
public class EventScene {


    @Getter
    private final boolean all;
    @Getter
    private final String name;

    private static final Logger log = LoggerFactory.getLogger("EventTrace");

    private static final List<EventScene> instances = new CopyOnWriteArrayList<>();

    public static final EventScene OTHER = valueOf("OTHER", false);

    public static EventScene valueOf(String name, boolean all) {
        for (EventScene eventScene : instances) {
            if (eventScene.getName().equals(name)) {
                return eventScene;
            }
        }

        synchronized (EventScene.class) {
            for (EventScene eventScene : instances) {
                if (eventScene.getName().equals(name)) {
                    return eventScene;
                }
            }
            EventScene eventScene = new EventScene(name, all);
            instances.add(eventScene);
            return eventScene;
        }
    }

    private EventScene(String name, boolean all) {
        this.all = all;
        this.name = name;
        for (int i = 0; i < slots.length; i++) {
            slots[i] = new Slot();
        }
    }

    private final Slot[] slots = new Slot[30];

    public static EventScene[] values() {
        return instances.toArray(new EventScene[]{});
    }

    private static class Slot {
        private final AtomicLong time = new AtomicLong(0);
        private Recorder recorder = nopRecorder;
    }

    private static final Recorder nopRecorder = new Recorder("none") {
        @Override
        public void recordEvent(MessageGetter messageGetter, Throwable throwable) {

        }
    };

    private static final Recorder otherRecorder = new RecorderImpl("other", OTHER);


    private static final Looper looper = new Looper("eventRecorder").startLoop();
    private LinkedList<WeakReference<RecorderImpl>> historyRecorders = new LinkedList<>();

    public static void post(Runnable runnable) {
        looper.post(runnable);
    }

    Map<String, List<Event>> fetchEvents() {
        looper.checkLooper();
        Map<String, List<Event>> ret = new HashMap<>();
        for (Slot slot : slots) {
            Recorder recorder = slot.recorder;
            if (!(recorder instanceof RecorderImpl)) {
                continue;
            }
            RecorderImpl recorderImpl = (RecorderImpl) recorder;
            ArrayList<Event> events = new ArrayList<>(recorderImpl.events);
            ret.put(recorderImpl.sessionId, events);
        }
        for (WeakReference<RecorderImpl> reference : historyRecorders) {
            RecorderImpl recorderImpl = reference.get();
            if (recorderImpl != null) {
                ArrayList<Event> events = new ArrayList<>(recorderImpl.events);
                ret.put(recorderImpl.sessionId, events);
            }
        }
        return ret;
    }


    public Recorder acquireRecorder(String sessionId, boolean debug) {
        if (debug || isAll()) {
            return new RecorderImpl(sessionId, this);
        }
        long nowTime = System.currentTimeMillis();
        int slotIndex = ((int) ((nowTime / 1000) % 60)) / 2;
        long timeMinute = nowTime / 60000;

        Slot slot = slots[slotIndex];

        long slotTime = slot.time.get();
        if (slotTime == timeMinute) {
            return nopRecorder;
        }
        if (slot.time.compareAndSet(slotTime, timeMinute)) {
            cacheRecordImpl(slot.recorder);
            slot.recorder = new RecorderImpl(sessionId, this);
            return slot.recorder;
        }
        return nopRecorder;
    }

    private static class RecorderImpl extends Recorder {
        private final LinkedList<Event> events = new LinkedList<>();
        private final String sessionId;
        private final EventScene eventScene;

        RecorderImpl(String sessionId, EventScene eventScene) {
            super(sessionId);
            this.sessionId = sessionId;
            this.eventScene = eventScene;
        }

        private Collection<String> splitMsg(String msg, Throwable throwable) {
            Collection<String> strings = StringSplitter.split(msg, '\n');
            if (throwable == null) {
                return strings;
            }
            if (strings.isEmpty()) {
                // 确保可以被编辑
                strings = new LinkedList<>();
            }
            ThrowablePrinter.printStackTrace(strings, throwable);
            return strings;
        }


        @Override
        public void recordEvent(MessageGetter messageGetter, Throwable throwable) {
            looper.post(() -> {
                MDC.put("Scene", eventScene.getName());
                String subTitle = getSubTitle();
                String message = messageGetter.getMessage();

                Collection<String> msgLines = splitMsg(messageGetter.getMessage(), throwable);
                for (String line : msgLines) {
                    if (subTitle != null) {
                        log.info("sessionId:{} subTitle:{} -> {}", sessionId, subTitle, line);
                    } else {
                        log.info("sessionId:{} -> {}", sessionId, line);
                    }
                }
                Event event = new Event(System.currentTimeMillis(), message, throwable);
                events.addLast(event);
            });
        }

        @Override
        public boolean enable() {
            return true;
        }
    }


    public static Recorder nop() {
        return nopRecorder;
    }

    public static Recorder other() {
        return otherRecorder;
    }

    private void cacheRecordImpl(Recorder recorder) {
        if (!(recorder instanceof RecorderImpl)) {
            return;
        }
        looper.post(() -> appendRecordImplImpl((RecorderImpl) recorder));
    }

    private void appendRecordImplImpl(RecorderImpl recorder) {
        if (!looper.inLooper()) {
            looper.post(() -> appendRecordImplImpl(recorder));
            return;
        }
        // 一轮 30个，一分钟
        // 10轮 300个，10分钟
        // session 流程记录最大考虑记录10分钟即可,那么cache最多需要300即可

        // 20210629 300-> 150，trace日志下放到文件中
        if (historyRecorders.size() > 150) {
            LinkedList<WeakReference<RecorderImpl>> newLinkList = new LinkedList<>();
            for (WeakReference<RecorderImpl> reference : historyRecorders) {
                RecorderImpl recorderImpl = reference.get();
                if (recorderImpl == null) {
                    // has been GC
                    continue;
                }
                newLinkList.add(new WeakReference<>(recorderImpl));
                if (newLinkList.size() > 150) {
                    break;
                }
            }
            historyRecorders = newLinkList;
        }

        historyRecorders.addFirst(new WeakReference<>(recorder));

    }
}
