package com.vladsch.flexmark.ext.autolink.internal;

import com.vladsch.flexmark.ast.Text;
import com.vladsch.flexmark.ext.autolink.AutolinkExtension;
import com.vladsch.flexmark.parser.InlineParser;
import com.vladsch.flexmark.parser.InlineTextParserExtension;
import com.vladsch.flexmark.parser.Parser;
import com.vladsch.flexmark.util.ast.Node;
import com.vladsch.flexmark.util.sequence.BasedSequence;
import com.vladsch.flexmark.util.sequence.Range;

import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AutoLinkInlineParserExtensionBase implements InlineTextParserExtension {
    public static final Pattern EMAIL_AUTOLINK = Pattern.compile("(?:\\\\?[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-])+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)+", Pattern.MULTILINE);
    public static final Pattern AUTOLINK = Pattern.compile("[a-zA-Z][a-zA-Z0-9.+-]{1,31}://[^<>" + "\u0000-\u0020" + "]+", Pattern.MULTILINE);
    public static final Pattern WEBSITE_AUTOLINK = Pattern.compile("www\\.[^<>" + "\u0000-\u0020" + "]+", Pattern.MULTILINE);

    private final Pattern ignoredLinks;

    public AutoLinkInlineParserExtensionBase(final InlineParser inlineParser) {
        String ignoreLinks = AutolinkExtension.IGNORE_LINKS.getFrom(inlineParser.getDocument());
        ignoredLinks = ignoreLinks.isEmpty() ? null : Pattern.compile(ignoreLinks);
    }

    protected boolean isIgnoredLinkPrefix(final CharSequence url) {
        if (ignoredLinks != null) {
            Matcher matcher = ignoredLinks.matcher(url);
            return matcher.matches();
        }
        return false;
    }

    protected abstract Pattern getPattern();
    protected abstract Node getAutoLinkNode(BasedSequence linkUrl);

    @Override
    public Range getWantedRange(final BasedSequence text, final int textIndex) {
        Matcher matcher = getPattern().matcher(text);
        matcher.region(textIndex, text.length());

        if (matcher.find()) {
            int startIndex = matcher.start();
            int endIndex = matcher.end();

            if (startIndex > 0) {
                char c = text.charAt(startIndex - 1);

                if (Character.isUnicodeIdentifierPart(c) || c == '-' || c == '.' || c == '@' || c == ':') {
                    // restore index, false alarm
                    return null;
                }
            }

            if (!isIgnoredLinkPrefix(text.subSequence(startIndex, endIndex))) {
                // keep only trailing ) and } if matched by opening ones in the link
                int bracketStartOffset = startIndex;

                while (endIndex > startIndex) {
                    char c = text.charAt(endIndex - 1);
                    if (c == ')' || c == '}') {
                        // see if have matching ( { in linkUrl
                        String pending = String.valueOf(c);
                        ArrayList<Integer> offsets = new ArrayList<>();
                        offsets.add(endIndex);
                        int lastMatchedBracket = bracketStartOffset;

                        int pos = endIndex - 1;
                        while (pos > bracketStartOffset) {
                            c = text.charAt(pos - 1);
                            if (c == '(' && pending.charAt(0) == ')' || c == '{' && pending.charAt(0) == '}') {
                                // matched
                                pending = pending.substring(1);
                                int removed = offsets.remove(0);
                                if (lastMatchedBracket < removed) lastMatchedBracket = removed;
                                if (pending.isEmpty()) break;
                            } else if (c == ')' || c == '}') {
                                // another level
                                pending = String.format("%s%s", String.valueOf(c), pending);
                                offsets.add(0, pos);
                            } else if (!(c == '.' || c == ' ' || c == '\t' || c == ';' || c == ':')) {
                                // cut off trailing bracket truncation at this point
                                if (lastMatchedBracket < pos) lastMatchedBracket = pos;
                            }

                            pos--;
                        }

                        if (lastMatchedBracket < endIndex) {
                            endIndex = lastMatchedBracket;
                        } else {
                            // all matched, done
                            break;
                        }
                    } else if (c == '.' || c == ' ' || c == '\t' || c == ';' || c == ':') {
                        endIndex--;
                    } else {
                        break;
                    }
                }

                if (endIndex > startIndex) {
                    return Range.of(startIndex, endIndex);
                }
            }
        }
        return null;
    }

    @Override
    public void createNode(final Node block, final BasedSequence text, final Range range) {
        Node linkNode = getAutoLinkNode(text);
        linkNode.setCharsFromContent();

        Text contentNode = new Text(text);
        linkNode.appendChild(contentNode);

        block.appendChild(linkNode);
    }
}
