/*
 * Decompiled with CFR 0.152.
 */
package com.vlkan.rfos;

import com.vlkan.rfos.ByteCountingOutputStream;
import com.vlkan.rfos.Rotatable;
import com.vlkan.rfos.RotationCallback;
import com.vlkan.rfos.RotationConfig;
import com.vlkan.rfos.policy.RotationPolicy;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.zip.GZIPOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RotatingFileOutputStream
extends OutputStream
implements Rotatable {
    private static final Logger LOGGER = LoggerFactory.getLogger(RotatingFileOutputStream.class);
    private final RotationConfig config;
    private final List<RotationCallback> callbacks;
    private final List<RotationPolicy> writeSensitivePolicies;
    private volatile ByteCountingOutputStream stream;

    public RotatingFileOutputStream(RotationConfig config) {
        this.config = Objects.requireNonNull(config, "config");
        this.callbacks = new ArrayList<RotationCallback>(config.getCallbacks());
        this.writeSensitivePolicies = RotatingFileOutputStream.collectWriteSensitivePolicies(config.getPolicies());
        this.stream = this.open(null, config.getClock().now());
        this.startPolicies();
    }

    private static List<RotationPolicy> collectWriteSensitivePolicies(Set<RotationPolicy> policies) {
        ArrayList<RotationPolicy> writeSensitivePolicies = new ArrayList<RotationPolicy>();
        for (RotationPolicy policy : policies) {
            if (!policy.isWriteSensitive()) continue;
            writeSensitivePolicies.add(policy);
        }
        return writeSensitivePolicies;
    }

    private void startPolicies() {
        for (RotationPolicy policy : this.config.getPolicies()) {
            policy.start(this);
        }
    }

    private ByteCountingOutputStream open(RotationPolicy policy, Instant instant) {
        try {
            FileOutputStream fileOutputStream = new FileOutputStream(this.config.getFile(), this.config.isAppend());
            this.invokeCallbacks(callback -> callback.onOpen(policy, instant, fileOutputStream));
            long size = this.config.isAppend() ? this.config.getFile().length() : 0L;
            return new ByteCountingOutputStream(fileOutputStream, size);
        }
        catch (IOException error) {
            String message = String.format("file open failure {file=%s}", this.config.getFile());
            throw new RuntimeException(message);
        }
    }

    @Override
    public void rotate(RotationPolicy policy, Instant instant) {
        try {
            this.unsafeRotate(policy, instant);
        }
        catch (Exception error) {
            String message = String.format("rotation failure {instant=%s}", instant);
            RuntimeException extendedError = new RuntimeException(message, error);
            this.invokeCallbacks(callback -> callback.onFailure(policy, instant, null, extendedError));
        }
    }

    private synchronized void unsafeRotate(RotationPolicy policy, Instant instant) throws Exception {
        File rotatedFile;
        Objects.requireNonNull(instant, "instant");
        this.unsafeCheckStream();
        this.invokeCallbacks(callback -> callback.onTrigger(policy, instant));
        if (this.config.getFile().length() == 0L) {
            LOGGER.debug("empty file, skipping rotation {file={}}", (Object)this.config.getFile());
            return;
        }
        this.invokeCallbacks(callback -> callback.onClose(policy, instant, this.stream));
        this.stream.close();
        if (this.config.getMaxBackupCount() > 0) {
            this.renameBackups();
            rotatedFile = this.backupFile();
        } else {
            rotatedFile = this.config.getFilePattern().create(instant).getAbsoluteFile();
            LOGGER.debug("renaming {file={}, rotatedFile={}}", (Object)this.config.getFile(), (Object)rotatedFile);
            RotatingFileOutputStream.renameFile(this.config.getFile(), rotatedFile);
        }
        LOGGER.debug("re-opening file {file={}}", (Object)this.config.getFile());
        this.stream = this.open(policy, instant);
        if (this.config.isCompress()) {
            this.asyncCompress(policy, instant, rotatedFile);
            return;
        }
        this.invokeCallbacks(callback -> callback.onSuccess(policy, instant, rotatedFile));
    }

    private void renameBackups() throws IOException {
        File dstFile = this.getBackupFile(this.config.getMaxBackupCount() - 1);
        for (int backupIndex = this.config.getMaxBackupCount() - 2; backupIndex >= 0; --backupIndex) {
            File srcFile = this.getBackupFile(backupIndex);
            if (srcFile.exists()) {
                LOGGER.debug("renaming backup {srcFile={}, dstFile={}}", (Object)srcFile, (Object)dstFile);
                RotatingFileOutputStream.renameFile(srcFile, dstFile);
            }
            dstFile = srcFile;
        }
    }

    private File backupFile() throws IOException {
        File dstFile = this.getBackupFile(0);
        File srcFile = this.config.getFile();
        LOGGER.debug("renaming for backup {srcFile={}, dstFile={}}", (Object)srcFile, (Object)dstFile);
        RotatingFileOutputStream.renameFile(srcFile, dstFile);
        return dstFile;
    }

    private static void renameFile(File srcFile, File dstFile) throws IOException {
        Files.move(srcFile.toPath(), dstFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private File getBackupFile(int backupIndex) {
        String parent = this.config.getFile().getParent();
        if (parent == null) {
            parent = ".";
        }
        String fileName = this.config.getFile().getName() + '.' + backupIndex;
        return Paths.get(parent, fileName).toFile();
    }

    private void asyncCompress(final RotationPolicy policy, final Instant instant, final File rotatedFile) {
        this.config.getExecutorService().execute(new Runnable(){
            private final String displayName;
            {
                this.displayName = String.format("%s.compress(%s)", RotatingFileOutputStream.class.getSimpleName(), rotatedFile);
            }

            @Override
            public void run() {
                File compressedFile = RotatingFileOutputStream.this.getCompressedFile(rotatedFile);
                try {
                    RotatingFileOutputStream.unsafeSyncCompress(rotatedFile, compressedFile);
                    RotatingFileOutputStream.this.invokeCallbacks(callback -> callback.onSuccess(policy, instant, compressedFile));
                }
                catch (Exception error) {
                    String message = String.format("compression failure {instant=%s, rotatedFile=%s, compressedFile=%s}", instant, rotatedFile, compressedFile);
                    RuntimeException extendedError = new RuntimeException(message, error);
                    RotatingFileOutputStream.this.invokeCallbacks(callback -> callback.onFailure(policy, instant, rotatedFile, extendedError));
                }
            }

            public String toString() {
                return this.displayName;
            }
        });
    }

    private File getCompressedFile(File rotatedFile) {
        String compressedFileName = String.format("%s.gz", rotatedFile.getAbsolutePath());
        return new File(compressedFileName);
    }

    private static void unsafeSyncCompress(File rotatedFile, File compressedFile) throws IOException {
        LOGGER.debug("compressing {rotatedFile={}, compressedFile={}}", (Object)rotatedFile, (Object)compressedFile);
        try (FileInputStream sourceStream = new FileInputStream(rotatedFile);
             FileOutputStream targetStream = new FileOutputStream(compressedFile);
             GZIPOutputStream gzipTargetStream = new GZIPOutputStream(targetStream);){
            RotatingFileOutputStream.copy(sourceStream, gzipTargetStream);
        }
        LOGGER.debug("deleting old file {rotatedFile={}}", (Object)rotatedFile);
        boolean deleted = rotatedFile.delete();
        if (!deleted) {
            String message = String.format("failed deleting old file {rotatedFile=%s}", rotatedFile);
            throw new IOException(message);
        }
    }

    private static void copy(InputStream source, OutputStream target) throws IOException {
        int readByteCount;
        byte[] buffer = new byte[8192];
        while ((readByteCount = source.read(buffer)) > 0) {
            target.write(buffer, 0, readByteCount);
        }
    }

    @Override
    public RotationConfig getConfig() {
        return this.config;
    }

    @Override
    public synchronized void write(int b) throws IOException {
        this.unsafeCheckStream();
        long byteCount = this.stream.size() + 1L;
        this.notifyWriteSensitivePolicies(byteCount);
        this.stream.write(b);
    }

    @Override
    public synchronized void write(byte[] b) throws IOException {
        this.unsafeCheckStream();
        long byteCount = this.stream.size() + (long)b.length;
        this.notifyWriteSensitivePolicies(byteCount);
        this.stream.write(b);
    }

    @Override
    public synchronized void write(byte[] b, int off, int len) throws IOException {
        this.unsafeCheckStream();
        long byteCount = this.stream.size() + (long)len;
        this.notifyWriteSensitivePolicies(byteCount);
        this.stream.write(b, off, len);
    }

    private void notifyWriteSensitivePolicies(long byteCount) {
        for (int writeSensitivePolicyIndex = 0; writeSensitivePolicyIndex < this.writeSensitivePolicies.size(); ++writeSensitivePolicyIndex) {
            RotationPolicy writeSensitivePolicy = this.writeSensitivePolicies.get(writeSensitivePolicyIndex);
            writeSensitivePolicy.acceptWrite(byteCount);
        }
    }

    @Override
    public synchronized void flush() throws IOException {
        if (this.stream != null) {
            this.stream.flush();
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.stream == null) {
            return;
        }
        this.invokeCallbacks(callback -> callback.onClose(null, this.config.getClock().now(), this.stream));
        this.stopPolicies();
        this.stream.close();
        this.stream = null;
    }

    private void stopPolicies() {
        this.config.getPolicies().forEach(RotationPolicy::stop);
    }

    private void invokeCallbacks(Consumer<RotationCallback> invoker) {
        for (int callbackIndex = 0; callbackIndex < this.callbacks.size(); ++callbackIndex) {
            RotationCallback callback = this.callbacks.get(callbackIndex);
            invoker.accept(callback);
        }
    }

    private void unsafeCheckStream() throws IOException {
        if (this.stream == null) {
            throw new IOException("either closed or not initialized yet");
        }
    }

    public String toString() {
        return String.format("RotatingFileOutputStream{file=%s}", this.config.getFile());
    }
}

