#ifndef ML_TRANSFORMERS_ML_TRANSFORMERS_H_
#define ML_TRANSFORMERS_ML_TRANSFORMERS_H_

#include <modules/vonage/api/media_processor/media_processor.h>
#include <modules/vonage/api/ml_transformers/ml_transformer_base_config.h>

namespace webrtc {
class AudioFrame;
class VideoFrame;
}

namespace vonage {


/** Background effect type.

    Enumerates all available background effects.
 */
enum class BackgroundEffectType {
    kBlur, /**<  Blur background. */
    kReplace, /**< Virtual background. */
};

/** Blur level enumeration listing all possible levels.

    Enumerates all blur levels possible for blur-related transformer instances.
 */
enum class BlurRadius {
    kNone, /**<   No blur level specified. */
    kHigh, /**<   High blur level. */
    kLow,/**<   Low blur level. */
    kCustom, /**< Custom blur level depends on user settings*/
};

/** Base struct for ML background effect transformer configuration.

 */
class BaseBackgroundEffectConfig {
public:
    BaseBackgroundEffectConfig(BackgroundEffectType type);
    virtual ~BaseBackgroundEffectConfig() = default;
    BackgroundEffectType getType() const;
    
private:
    BackgroundEffectType type_; /**<  Background effect type enumeration value. */
};

/** ML background blur transformer configuration struct.
When using kCustom radius custom_radius must be provided.
When running on GPU Bilateral Filter is used (See https://en.wikipedia.org/wiki/Bilateral_filter).
When running on CPU Box Blur is used (See https://en.wikipedia.org/wiki/Box_blur).
Please consider this when using (custom) user-provided blur level.
 */
class BlurBackgroundEffectConfig : public BaseBackgroundEffectConfig {
public:
    BlurBackgroundEffectConfig(BlurRadius radius, uint8_t custom_radius = 0);
    virtual ~BlurBackgroundEffectConfig() = default;
    BlurRadius GetRadius() const;
    uint8_t GetCustomRadius() const;
private:
    BlurRadius radius_; /**<  Blur level enumeration value. */
    uint8_t    custom_radius_; /**< Blur level to be used as radius when (custom) user-provided value is selected. */
};

/** ML background replacement (virtual background) transformer configuration struct.

 */
class ReplaceBackgroundEffectConfig : public BaseBackgroundEffectConfig {
public:
    ReplaceBackgroundEffectConfig(const std::string& image_file_path);
    ReplaceBackgroundEffectConfig(const std::wstring& image_file_path);

    virtual ~ReplaceBackgroundEffectConfig() = default;
    const std::string& GetFilePath() const;
private:
    std::string image_file_path_; /**< Path to the image resource. Supported formats are: PNG, TIFF and JPEG. */
};


/**< Video background effect transformer configuration class.
See example about how to instantiate a new config object instance below.
@code
 std::vector<std::string> ml_model_file_paths;
 ml_model_file_paths.push_back(selfie_segmentation_model_file_path);
 std::unique_ptr<BaseBackgroundEffectConfig> background_config;
 if(image_backgound_file_path.empty()){
    background_config = std::make_unique<BlurBackgroundEffectConfig>(BlurRadius::kHigh);
 }else{
    background_config = std::make_unique<ReplaceBackgroundEffectConfig>(image_backgound_file_path);
 }
 std::unique_ptr<MLTransformerBaseConfig> config = std::make_unique<MLTransformerVideoBackgroundEffectConfig>(ml_model_files_path, std::move(background_config));
@endcode
 */
class MLTransformerVideoBackgroundEffectConfig : public MLTransformerBaseConfig {
public:
    MLTransformerVideoBackgroundEffectConfig(const std::vector<std::string>& tflite_model_paths, const BaseBackgroundEffectConfig* config);
    MLTransformerVideoBackgroundEffectConfig(const std::vector<std::wstring>& tflite_model_paths, const BaseBackgroundEffectConfig* config);

    virtual ~MLTransformerVideoBackgroundEffectConfig() = default;
    const BaseBackgroundEffectConfig& GetConfig() const;
protected:
    const BaseBackgroundEffectConfig* config_;
};

/** Vonage video transformer helper class with factory function.

 */
class VonageVideoTransformer {
public:
    /** Create factory function.

     Creates transformer instances according to the config object set. An observer instance can be provided so the application can observer transformer events.
     */
    static std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::VideoFrame>> Create(const MLTransformerBaseConfig* config, webrtc::BaseFrameTransformerObserver* observer);
};

} // namespace vonage

#endif // ML_TRANSFORMERS_ML_TRANSFORMERS_H_
